provider "aws" {
    profile = "${var.main_account}"
    region  = "${var.region}"
}

resource "aws_s3_bucket" "bucket" {
    bucket = "${data.null_data_source.config.inputs.bucket_name}"
    acl    = "private"

    versioning {
        enabled = true
    }

    tags {
        Name  = "${data.null_data_source.config.inputs.bucket_name} - CodeDeploy App Bucket"
        Owner = "${var.owner}@twitch.tv"
    }
}

data "aws_iam_policy_document" "deployer_policy_doc" {
    statement {
        actions = [
            "codedeploy:List*",
            "codedeploy:Get*",
            "codedeploy:CreateDeployment",
            "codedeploy:RegisterApplicationRevision"
        ]

        resources = ["*"]
    }
}

resource "aws_iam_policy" "deployer_policy" {
    name = "${data.null_data_source.config.inputs.bucket_name}-actor-deploy"
    policy = "${data.aws_iam_policy_document.deployer_policy_doc.json}"
}

resource "aws_iam_user" "deployer" {
    name = "${data.null_data_source.config.inputs.bucket_name}-actor"
}


resource "aws_iam_user_policy_attachment" "deployer_policy_attachment" {
    user = "${aws_iam_user.deployer.name}"
    policy_arn = "${aws_iam_policy.deployer_policy.arn}"
}

data "aws_iam_policy_document" "bucket_policy_doc" {
    statement {
        actions = [
            "s3:Put*",
            "s3:ListBucket",
            "s3:GetObject"
        ]

        principals {
            type = "AWS"
            identifiers = [
                "${aws_iam_user.deployer.arn}"
            ]
        }

        resources = [
            "arn:aws:s3:::${data.null_data_source.config.inputs.bucket_name}",
            "arn:aws:s3:::${data.null_data_source.config.inputs.bucket_name}/*"
        ]
    }

    statement {
        actions = [
            "s3:GetObject",
            "s3:ListBucket",
            "s3:GetBucketLocation"
        ]

        principals {
            type = "AWS"
            identifiers = ["${var.secondary_accounts}"]
        }

        resources = [
            "arn:aws:s3:::${data.null_data_source.config.inputs.bucket_name}",
            "arn:aws:s3:::${data.null_data_source.config.inputs.bucket_name}/*"
        ]
    }
}

resource "aws_s3_bucket_policy" "bucket_policy" {
    bucket = "${aws_s3_bucket.bucket.id}"
    policy = "${data.aws_iam_policy_document.bucket_policy_doc.json}"
}

output "main_data" {
    value = {
        region = "${var.region}"
        main_account = "${var.main_account}"
        deployer_user = "${aws_iam_user.deployer.name}"
        deployer_user_arn = "${aws_iam_user.deployer.arn}"
        bucket_name = "${data.null_data_source.config.inputs.bucket_name}"
    }
}

