import math

import numpy as np
import matplotlib
from matplotlib.font_manager import FontProperties
from matplotlib.ticker import FuncFormatter
import psycopg2

from config.settings import *
from settings import *
import util.db
from util.log import log
import reports.times
from util.matplotlib import magnitude_ticks
from matplotlib import dates

matplotlib.use('Agg')


def checkpoint_activity():
    import matplotlib.pyplot as plot

    db = util.db.connect()
    
    cur = db.cursor()
    
    cur.execute("""
        SELECT second_tick AS measurement_time,
               COALESCE(bytes_written / EXTRACT(EPOCH FROM (checkpoint_end-checkpoint_start)), 0) AS bytes_per_second
            FROM (
                    SELECT * FROM generate_series(%(start_time)s, %(end_time)s, '1 second'::interval)
                ) log_times(second_tick)
            LEFT OUTER JOIN checkpoints
                ON second_tick BETWEEN checkpoint_start AND checkpoint_end
            ORDER BY measurement_time
            """, reports.times.log_range())
    
    results = cur.fetchall()

    if not results:
        no_file = REPORT_DIR_PATH + '/checkpoint_activity.txt'
        with open(no_file, 'w') as no_results:
            no_results.write("No results.")
        no_results.close()
        log('No checkpoints recorded; skipping checkpoint_activity report')
        return
    
    timeseries = [r[0] for r in results]
    bytes_written = [r[1] for r in results]
    
    fig = plot.figure(figsize=DEFAULT_FIGURE_SIZE)
    ax = fig.add_subplot(111, axisbg='#fafafa', alpha=0.9)
    hfmt = dates.DateFormatter('%H:%M:%S')

    ax.set_title('Checkpoint Activity')
    ax.set_ylabel('Bytes/Second Written')
    ax.xaxis.set_major_formatter(hfmt)
    ax.set_xlim(xmin=reports.times.log_start(), xmax=reports.times.log_end())
    ax.yaxis.set_major_formatter(FuncFormatter(magnitude_ticks))
    ax.grid(True)

    ax.fill_between(timeseries, 0, bytes_written, linewidth=0, color='red')

    fig.autofmt_xdate()
    
    plot.savefig(REPORT_DIR_PATH + '/checkpoint_activity.pdf')

    plot.close()

def checkpoint_detail():
    db = util.db.connect()
    cur = db.cursor()
    cur.execute("SELECT * from checkpoints_data()")

    if cur.rowcount > 0:
        reports.text_report(cur, REPORT_DIR_PATH + '/checkpoint_detail.csv')

    else:
        no_file = REPORT_DIR_PATH + '/checkpoint_detail.txt'
        with open(no_file, 'w') as no_results:
            no_results.write("No results.")
        no_results.close()
        log('No connections recorded; skipping checkpoint_detail report')
        return
