import numpy as np
import matplotlib
from matplotlib.font_manager import FontProperties
from matplotlib.ticker import FormatStrFormatter, FuncFormatter
from util.matplotlib import magnitude_ticks
import psycopg2

from config.settings import *
from settings import *
import util.db
import reports.times
from util import table_scale
from util.log import log


matplotlib.use('Agg')
    
def temp_files():
    
    import matplotlib.pyplot as plot
    
    db = util.db.connect()
    
    small_font = FontProperties()
    small_font.set_size('x-small')
    
    db = util.db.connect()
    
    cur = db.cursor()
    
    cur.execute("""
        SELECT pg_size_pretty(magnitude::decimal), file_count FROM (
            SELECT
                CASE WHEN file_size < 10             THEN    1
                     WHEN file_size < 100            THEN   10
                     WHEN file_size < 1024           THEN  100
                     when file_size < 1024*10        THEN 1024
                     WHEN file_size < 1024*100       THEN 1024*10
                     WHEN file_size < 1024*1024      THEN 1024*100
                     WHEN file_size < 1024*1024*10   THEN 1024*1024
                     WHEN file_size < 1024*1024*100  THEN 1024*1024*10
                     WHEN file_size < 1024*1024*1024 THEN 1024*1024*100
                     ELSE 1024*1024*1024 END AS magnitude,
                COUNT(*) as file_count
                FROM temp_files
                GROUP BY 1
                ORDER BY magnitude
            ) AS mags
        """)
    
    results = cur.fetchall()

    if not results:
        no_file = REPORT_DIR_PATH + '/temp_files.txt'
        with open(no_file, 'w') as no_results:
            no_results.write("No results.")
        no_results.close()
        log('No temp files recorded; skipping temp_files report')
        return

        
    counts = [ i[1] for i in results ]
    labels = [ i[0] for i in results ]
    
    ind = np.arange(len(counts))
    width = 0.9
    
    fig = plot.figure(figsize=DEFAULT_FIGURE_SIZE)
    ax = fig.add_subplot(111)
    
    ax.set_title('Temporary Files')
    ax.set_ylabel('Count')
    ax.set_xlabel('Temporary File Size (in bytes, order of magnitude)')
    
    ax.set_xticks(ind + width/2)
    ax.set_xticklabels(labels, fontproperties=small_font)

    ax.bar(ind, counts, width, color='r')
    
    plot.savefig(REPORT_DIR_PATH + '/temp_files.pdf')

    plot.close()


def temp_files_by_time():
    import matplotlib.pyplot as plot
    
    x_small_font = FontProperties()
    x_small_font.set_size('x-small')
    
    db = util.db.connect()
    
    cur = db.cursor()
    
    cur.execute("""
        SELECT date_trunc('minute', measurement_time) AS mt, 
               MAX(file_size),
               AVG(file_size),
               SUM(file_size)
            FROM temp_files
            WHERE measurement_time BETWEEN %(start_time)s AND %(end_time)s
            GROUP BY 1
            ORDER BY mt
        """, reports.times.log_range())
    
    results = cur.fetchall()
    
    if not results:
        no_file = REPORT_DIR_PATH + '/temp_files_by_time.txt'
        with open(no_file, 'w') as no_results:
            no_results.write("No results.")
        no_results.close()
        log('No temp files recorded; skipping temp_files_by_time report')
        return
        
    timeseries = [r[0] for r in results]
    max_filesize = [float(r[1]) for r in results]
    avg_filesize = [float(r[2]) for r in results]
    sum_filesize = [float(r[3]) for r in results]
        
    fig = plot.figure(figsize=DEFAULT_FIGURE_SIZE)
    ax = fig.add_subplot(111, axisbg='#fafafa', alpha=0.9)
    
    ax.set_title('Temporary Files By Minute')
    ax.set_ylabel('File Size (log scale)')
    
    ax.set_yscale('log')
    ax.set_xlim(xmin=reports.times.log_start(), xmax=reports.times.log_end())
    ax.grid(True)
    
    ax.yaxis.set_major_formatter(FuncFormatter(magnitude_ticks))
    
    ax.scatter(timeseries, avg_filesize, s=4, color='blue', label='Average File Size')
    ax.scatter(timeseries, sum_filesize, s=4, color='red', label='Total Temp Storage')
    ax.scatter(timeseries, max_filesize, s=4, color='orange', label='Maximum File Size')
    fig.autofmt_xdate()
    
    lgnd = ax.legend(loc='upper right', bbox_to_anchor=(1.25, 1.0), prop=x_small_font)

    plot.tight_layout()
    
    plot.savefig(REPORT_DIR_PATH + '/temp_files_by_time.pdf', bbox_inches='tight', pad_inches=0.25, bbox_extra_artists=[lgnd])

    plot.close()

def temp_files_by_time_bar():
    import matplotlib.pyplot as plot

    x_small_font = FontProperties()
    x_small_font.set_size('x-small')

    db = util.db.connect()

    cur = db.cursor()

    cur.execute("""
        SELECT date_trunc('minute', measurement_time) AS mt,
               MAX(file_size),
               AVG(file_size),
               SUM(file_size)
            FROM temp_files
            WHERE measurement_time BETWEEN %(start_time)s AND %(end_time)s
            GROUP BY 1
            ORDER BY mt
        """, reports.times.log_range())

    results = cur.fetchall()

    if not results:
        no_file = REPORT_DIR_PATH + '/temp_files_by_time.txt'
        with open(no_file, 'w') as no_results:
            no_results.write("No results.")
        no_results.close()
        log('No temp files recorded; skipping temp_files_by_time report')
        return

    timeseries = [r[0] for r in results]
    max_filesize = [float(r[1]) for r in results]
    avg_filesize = [float(r[2]) for r in results]
    sum_filesize = [float(r[3]) for r in results]

    fig = plot.figure(figsize=DEFAULT_FIGURE_SIZE)
    ax = fig.add_subplot(111, axisbg='#fafafa', alpha=0.9)

    ax.set_title('Temporary Files By Minute')
    ax.set_ylabel('File Size (log scale)')

    ax.set_yscale('log')
    ax.set_xlim(xmin=reports.times.log_start(), xmax=reports.times.log_end())
    ax.grid(True)

    ax.yaxis.set_major_formatter(FuncFormatter(magnitude_ticks))

    ax.barh(timeseries, avg_filesize, linewidth=0, color='blue', label='Average File Size')
    ax.barh(timeseries, sum_filesize, linewidth=0, color='red', label='Total Temp Storage')
    ax.barh(timeseries, max_filesize, linewidth=0, color='orange', label='Maximum File Size')
    fig.autofmt_xdate()

    lgnd = ax.legend(loc='upper right', bbox_to_anchor=(1.25, 1.0), prop=x_small_font)

    plot.tight_layout()

    plot.savefig(REPORT_DIR_PATH + '/temp_files_by_time.pdf', bbox_inches='tight', pad_inches=0.25, bbox_extra_artists=[lgnd])

    plot.close()

