BEGIN;

DROP FUNCTION IF EXISTS create_connection_tables();

CREATE FUNCTION create_connection_tables() RETURNS VOID AS
$$
BEGIN

    drop table IF EXISTS connections;
    drop table IF EXISTS disconnections;
    
    create table connections as 
    select log_time, session_id, session_line_num, user_name, database_name,
        substring(connection_from from $x$^(.*):.*$x$) as client_host,
        substring(connection_from from $x$^.*:(.*)$x$) as port    
    from postgres_log
    where log_time between ( select log_start from time_boundaries )
            and ( select log_end from time_boundaries )
    and command_tag = 'authentication';
    
    create table disconnections as
    select log_time, session_id, session_line_num, user_name, database_name,
        substring(connection_from from $x$^(.*):.*$x$) as client_host,
        substring(connection_from from $x$^.*:(.*)$x$) as port,
        (substring(message from $x$disconnection: session time: ([\d:\.]+) user$x$))::interval
            as connection_time
    from postgres_log
    where log_time between ( select log_start from time_boundaries )
            and ( select log_end from time_boundaries )
    and message like 'disconnection: %';

END;
$$
    LANGUAGE 'plpgsql';


DROP FUNCTION IF EXISTS connections_by_minute();

CREATE FUNCTION connections_by_minute(OUT measurement_time TIMESTAMP WITH TIME ZONE,
                                      OUT connections NUMERIC) RETURNS SETOF RECORD AS
$connections_by_minute$
BEGIN

    RETURN QUERY
        select  minstart, 
                start_count + sum( conn_count - disc_count ) OVER ( order by minstart ) as conns
        from (
        select minstart,
                coalesce(conn_count,0) as conn_count,
                coalesce(disc_count,0) as disc_count
        from log_minutes
                left outer join
                ( select date_trunc('minute', log_time) as contime,
                        count(*) as conn_count
                        from connections
                        group by 1 ) as conns
                on minstart = conns.contime
                left outer join
                ( select date_trunc('minute', log_time) as contime,
                        count(*) as disc_count
                        from disconnections
                        group by 1 ) as disconns
                on minstart = disconns.contime
        ) as connects,
        ( select count(*) as start_count from server.pg_stat_activity_start ) as start_connects;

END;
$connections_by_minute$
    LANGUAGE 'plpgsql';

COMMIT;