package migrations

import (
	"errors"
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"
	"time"
)

type MigrationSource interface {
	FetchAllMigrations() ([]*MigrationRecord, error)
	FetchSingleMigration(name string) (*MigrationRecord, error)
}

type DirectoryMigrationSource struct {
	MigrationsDirectory string
}

func (source *DirectoryMigrationSource) FetchAllMigrations() ([]*MigrationRecord, error) {
	var m []*MigrationRecord

	exists, err := directoryExists(source.MigrationsDirectory)
	if err != nil {
		return m, err
	}
	if !exists {
		return m, nil
	}

	err = filepath.Walk(source.MigrationsDirectory, func(name string, info os.FileInfo, e error) error {
		if !strings.HasSuffix(info.Name(), ".yaml") {
			return nil
		}
		data, e := ioutil.ReadFile(filepath.Join(source.MigrationsDirectory, info.Name()))
		if e != nil {
			return e
		}
		creationTime, e := extractCreationTimeFromFile(info.Name())
		if e != nil {
			return e
		}
		m = append(m, &MigrationRecord{
			info.Name(),
			extractNameFromFile(info.Name()),
			creationTime,
			data,
		})
		return nil
	})

	if err != nil {
		return m, err
	}

	return m, nil
}

func (source *DirectoryMigrationSource) FetchSingleMigration(name string) (*MigrationRecord, error) {
	data, err := ioutil.ReadFile(filepath.Join(source.MigrationsDirectory, name))
	if err != nil {
		return nil, errors.New("Last applied migration not found. Is your local directory up-to-date?")
	}

	creationTime, err := extractCreationTimeFromFile(name)
	if err != nil {
		return nil, err
	}

	return &MigrationRecord{
		name,
		extractNameFromFile(name),
		creationTime,
		data,
	}, nil
}

// helper func to get name of migration from filename
func extractNameFromFile(name string) string {
	s := strings.Split(name, "_")
	s[len(s)-1] = strings.Split(s[len(s)-1], ".")[0]

	return strings.Join(s[1:], " ")
}

// helper func to get creation time from filename
func extractCreationTimeFromFile(name string) (time.Time, error) {
	s := strings.Split(name, "_")
	t, err := time.Parse("20060102150405", s[0])
	if err != nil {
		return time.Now(), err
	}
	return t, nil
}
