package main

import (
	"encoding/csv"
	"log"
	"os"
	"sort"
	"strconv"
	"strings"

	"code.justin.tv/common/config"
	"code.justin.tv/release/trace/pglex"
	"code.justin.tv/release/trace/pgtables"
)

var tableNames []string

func main() {
	config.Register(map[string]string{
		"File":    "",
		"OutFile": "./output.csv",
	})

	err := config.Parse()
	if err != nil {
		log.Fatal(err)
	}

	filePath := config.MustResolve("File")
	outFilePath := config.MustResolve("OutFile")

	file, err := os.Open(filePath)
	if err != nil {
		log.Fatalln(err)
	}

	reader := csv.NewReader(file)
	records, err := reader.ReadAll()
	if err != nil {
		log.Fatalln(err)
	}

	irrelevantCommands := map[string]bool{
		"BEGIN":       true,
		"COMMIT":      true,
		"DISCARD ALL": true,
		"ROLLBACK":    true,
		"SET":         true,
		"SHOW":        true,
		"VACUUM":      true,
	}

	results := map[string]int{}

	for _, record := range records {
		if len(record) > 13 {
			if record[12] != "00000" {
				continue
			}
			if strings.HasPrefix(record[7], "idle") {
				continue
			}
			if strings.HasPrefix(record[1], "backup") {
				continue
			}
			if _, ignore := irrelevantCommands[record[7]]; ignore {
				continue
			}

			queryData := record[13]
			query := retrieveQuery(queryData)

			tables, err := pgtables.Cache.Tables(query)
			if err != nil {
				log.Println(query)
				log.Println(err)
				continue
			}
			tableStr := buildTableStr(tables)
			count, ok := results[tableStr]
			if !ok {
				count = 0
			}
			results[tableStr] = count + 1
		}
	}

	outFile, err := os.Create(outFilePath)
	if err != nil {
		log.Fatalln(err)
	}
	writer := csv.NewWriter(outFile)

	for key, val := range results {
		if key == "" {
			continue
		}

		countStr := strconv.Itoa(val)
		err = writer.Write([]string{key, countStr})
		if err != nil {
			log.Fatalln(err)
		}
	}

	writer.Flush()
}

func retrieveQuery(queryData string) string {
	queryBegin := strings.Index(queryData, "duration:")
	if queryBegin >= 0 {
		queryBegin += 8
		queryData = queryData[queryBegin+1:]
	}

	queryBegin = strings.Index(queryData, ":")
	if queryBegin >= 0 {
		query := queryData[queryBegin+1:]
		return processQuery(query)
	}

	return processQuery(queryData)
}

func buildTableStr(tables []string) string {
	sort.Strings(tables)
	return strings.Join(tables, ",")
}

func processQuery(rawQuery string) string {
	rawQuery = strings.TrimRight(strings.TrimSpace(rawQuery), ";")
	processed := pglex.Normalize(rawQuery)
	if strings.Contains(processed, "lex error") {
		processed = processed + " - " + rawQuery
	}

	return processed
}
