package comparison

import (
	"encoding/json"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

type TestType2 struct {
	Field []int
}

type TestType1 struct {
	Test  int
	Test2 string
	Test3 TestType2
}

func TestCreateFromJSON(t *testing.T) {
	Convey("Comparable from JSON", t, func() {
		mapVal := make([]interface{}, 0)
		err := json.Unmarshal([]byte(`[
            "test",
            3,
            {
                "field": [
                    1,
                    2,
                    3
                ]
            }
        ]`), &mapVal)

		So(err, ShouldBeNil)

		cmp := ComparablesFromJSONObj(mapVal)
		So(cmp[0].NextResultFromPath(make([]string, 0)), ShouldEqual, "test")
		So(cmp[1].NextResultFromPath(make([]string, 0)), ShouldEqual, 3)
		So(cmp[2].NextResultFromPath([]string{
			"field",
			"1",
		}), ShouldEqual, 2)
		So(cmp[2].NextResultFromPath([]string{
			"field",
			"*",
		}), ShouldEqual, 1)
		So(cmp[2].NextResultFromPath([]string{
			"field",
			"*",
		}), ShouldEqual, 3)
		So(cmp[2].NextResultFromPath([]string{
			"field",
			"*",
		}), ShouldBeNil)

		mapVal2 := make([]interface{}, 0)
		err = json.Unmarshal([]byte(`[
            "test",
            3,
            {
                "field": []
            }
        ]`), &mapVal2)
		So(err, ShouldBeNil)

		cmp2 := ComparablesFromJSONObj(mapVal2)

		same, err := DeepCompare("test", nil, cmp, cmp2)
		So(err, ShouldBeNil)
		So(same, ShouldBeTrue)
	})
}

func TestCreateFromStruct(t *testing.T) {
	Convey("Comparable from struct", t, func() {
		val := TestType1{
			Test:  3,
			Test2: "test",
			Test3: TestType2{
				Field: []int{1, 2, 3},
			},
		}

		cmp, err := ComparablesFromStruct([]interface{}{val})
		So(err, ShouldBeNil)

		So(cmp[0].NextResultFromPath([]string{
			"Test",
		}), ShouldEqual, 3)

		So(cmp[0].NextResultFromPath([]string{
			"Test",
		}), ShouldBeNil)

		So(cmp[0].NextResultFromPath([]string{
			"Test2",
		}), ShouldEqual, "test")

		So(cmp[0].NextResultFromPath([]string{
			"Test2",
		}), ShouldBeNil)

		So(cmp[0].NextResultFromPath([]string{
			"Test3",
			"Field",
			"1",
		}), ShouldEqual, 2)

		So(cmp[0].NextResultFromPath([]string{
			"Test3",
			"Field",
			"*",
		}), ShouldEqual, 1)

		So(cmp[0].NextResultFromPath([]string{
			"Test3",
			"Field",
			"*",
		}), ShouldEqual, 3)

		So(cmp[0].NextResultFromPath([]string{
			"Test3",
			"Field",
			"*",
		}), ShouldBeNil)

		mapVal2 := make([]interface{}, 0)
		err = json.Unmarshal([]byte(`[
            {
                "Test3": {
                    "Field": []
                }
            }
        ]`), &mapVal2)
		So(err, ShouldBeNil)

		cmp2 := ComparablesFromJSONObj(mapVal2)

		same, err := DeepCompare("test", nil, cmp, cmp2)
		So(err, ShouldBeNil)
		So(same, ShouldBeTrue)
	})
}
