package migration

import (
	"encoding/json"
	"fmt"
	"log"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/kinesis"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/chat/timing"
)

var StatsdSampleRate float32 = 0.1

type EventWriter interface {
	Write(method string, params map[string]interface{}, results []interface{}, duration time.Duration)
}

type PartitionKeyBuilder interface {
	PartitionKey(method string, params map[string]interface{}, results []interface{}) string
}

func CreateKinesisWriter(statsdHostPort string, environment string, repo string, interfaceName string, partitionKey PartitionKeyBuilder, awsProfile string, region string, streamName string) EventWriter {
	stats, err := statsd.NewBufferedClient(statsdHostPort,
		fmt.Sprintf("d8a-migration.%s.%s.%s", environment, extractRepoFromSvcname(repo), interfaceName),
		time.Second, 0)
	if err != nil {
		log.Fatalln(err)
	}

	awsConfig := &aws.Config{
		Credentials: getCredentialsFromProfile(awsProfile),
		Region:      aws.String(region),
	}
	sess := session.New()

	return &kinesisWriter{
		stream:       kinesis.New(sess, awsConfig),
		partitionKey: partitionKey,
		streamName:   streamName,
		stats:        stats,
	}
}

type kinesisWriter struct {
	stream       *kinesis.Kinesis
	partitionKey PartitionKeyBuilder
	streamName   string
	stats        statsd.Statter
}

func (self *kinesisWriter) Write(method string, params map[string]interface{}, results []interface{}, duration time.Duration) {
	timer := timing.Xact{
		Stats:            self.stats,
		StatsdSampleRate: StatsdSampleRate,
	}

	timer.AddName(fmt.Sprintf("kinesis-write.%s", method))
	timer.Start()
	defer timer.End("err")

	data := map[string]interface{}{
		"Exp__Method":   method,
		"Exp__Duration": duration,
	}

	for key, val := range params {
		data[fmt.Sprintf("Param_%s", key)] = val
	}
	data["ReturnVals"] = results

	dataBytes, err := json.Marshal(data)
	if err != nil {
		log.Println(err)
		timer.End("err")
		return
	}

	partitionKey := self.partitionKey.PartitionKey(method, params, results)
	request := &kinesis.PutRecordInput{
		Data:         dataBytes,
		PartitionKey: &partitionKey,
		StreamName:   &self.streamName,
	}

	_, err = self.stream.PutRecord(request)
	if err != nil {
		log.Println(err)
		timer.End("err")
		return
	}

	timer.End("success")
}

func getCredentialsFromProfile(awsProfile string) *credentials.Credentials {
	if awsProfile == "" {
		return nil
	}

	return credentials.NewSharedCredentials("", awsProfile)
}

func extractRepoFromSvcname(svcname string) string {
	if svcname != "" {
		svcname = strings.Replace(svcname, "code.justin.tv/", "", -1)
		svcname = strings.Replace(svcname, "/", "-", -1)
	} else {
		svcname = "unknown"
	}

	return svcname
}
