package stats

import (
	"database/sql"
	"fmt"
	"log"
	"os"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/chat/timing"
	"code.justin.tv/d8a/pg-healthcheck/common"
)

var statsdSampleRate float32 = 1.0 //Sample all stats because we only need to write metrics once every few seconds

func NewStatsWriter(innerBackend common.BackendChecker, cluster string, port int, role RoleContainer, statter statsd.Statter) common.BackendChecker {
	return &statsWriter{
		statter:      statter,
		innerBackend: innerBackend,
		port:         port,
		role:         role,
		cluster:      cluster,
	}
}

type statsWriter struct {
	statter      statsd.Statter
	innerBackend common.BackendChecker
	role         RoleContainer
	dsn          string
	port         int
	cluster      string
}

func (sw *statsWriter) Check() (string, error) {
	if sw.role.Role() == "" {
		db, err := sw.Open()
		if err != nil {
			return "", err
		}
		defer common.CloseResource(db)

		err = sw.role.Refresh(db)
		if err == nil {
			sw.refreshPrefix()
			return "Healthy", nil
		}

		return "", err
	}

	metricName := fmt.Sprintf("health.%d", sw.port)
	timerResult := "ok"
	timer := timing.Xact{
		Stats:            sw.statter,
		StatsdSampleRate: statsdSampleRate,
	}

	timer.AddName(metricName)
	timer.Start()
	defer timer.End(timerResult)

	healthMsg, err := sw.innerBackend.Check()

	if err != nil {
		timerResult = "err"
		statError := sw.statter.Inc(metricName+".err", 1, statsdSampleRate)
		if statError != nil {
			log.Println(statError)
		}
	} else {
		statError := sw.statter.Inc(metricName+".success", 1, statsdSampleRate)
		if statError != nil {
			log.Println(statError)
		}
	}

	return healthMsg, err
}

func (sw *statsWriter) Open() (*sql.DB, error) {
	return sw.innerBackend.Open()
}

func (sw *statsWriter) refreshPrefix() {
	hostname, err := os.Hostname()
	if err != nil {
		hostname = "unknown"
	}

	prefix := fmt.Sprintf("pg-healthcheck.%s-%s.%s.%s", sw.cluster, os.Getenv("ENVIRONMENT"), sw.role.Role(), hostname)
	sw.statter.SetPrefix(prefix)
}
