package main

import (
	"errors"
	"flag"
	"fmt"
	"log"
	"strconv"
	"time"

	"database/sql"

	"github.com/cactus/go-statsd-client/statsd"
	_ "github.com/lib/pq"
)

// type used for flag.Var()
type intslice []int

type bouncer struct {
	dsn  string
	port int
}

var (
	cluster  string // eg, usherdb11 or db17.sfo01 or whatever
	server   string // host:port
	interval int

	host   string
	ports  intslice
	user   string
	pass   string
	schema string
)

func logRow(statsd statsd.Statter, port int, database, user, col, value string) {
	line := fmt.Sprintf("pgbouncer.%v.%v.%v.%v.%v.max", cluster, port, database, user, col)
	intVal, err := strconv.ParseInt(value, 10, 64)
	if err != nil {
		log.Println(err)
	}
	if err := statsd.Gauge(line, intVal, 1.0); err != nil {
		log.Println(err)
	}
}

// pools "schema"
// database  |   user    | cl_active | cl_waiting | sv_active | sv_idle | sv_used | sv_tested | sv_login | maxwait
//-----------+-----------+-----------+------------+-----------+---------+---------+-----------+----------+---------
// pgbouncer | pgbouncer |         1 |          0 |         0 |       0 |       0 |         0 |        0 |       0

func iterateRows(stats statsd.Statter, port int, rows *sql.Rows) error {
	columns, err := rows.Columns()
	if err != nil {
		return err
	}
	count := len(columns)
	for rows.Next() {
		var database, user string
		var cl_active, cl_waiting string
		var sv_active, sv_idle, sv_used, sv_tested, sv_login string
		var maxwait, maxwait_us, pool_mode string
		switch {
		case 10 == count:
			err = rows.Scan(&database, &user, &cl_active, &cl_waiting,
				&sv_active, &sv_idle, &sv_used, &sv_tested, &sv_login, &maxwait)
		case 11 == count:
			// pgbouncer 1.6+ picks up an 11th column
			// pool_mode
			//------------
			// transaction
			err = rows.Scan(&database, &user, &cl_active, &cl_waiting,
				&sv_active, &sv_idle, &sv_used, &sv_tested, &sv_login, &maxwait,
				&pool_mode)
		case 12 == count:
			// pgbouncer 1.8+ picks up a 12th column
			// maxwait_us
			err = rows.Scan(&database, &user, &cl_active, &cl_waiting,
				&sv_active, &sv_idle, &sv_used, &sv_tested, &sv_login, &maxwait,
				&maxwait_us, &pool_mode)
		case true:
			err = errors.New("Do not have usable schema for pools relation")
		}
		if err != nil {
			return err
		}
		if database == "pgbouncer" {
			// skip the pgbouncer database
			continue
		}
		logRow(stats, port, database, user, "cl_active", cl_active)
		logRow(stats, port, database, user, "cl_waiting", cl_waiting)
		logRow(stats, port, database, user, "sv_active", sv_active)
		logRow(stats, port, database, user, "maxwait", maxwait)
	}
	return nil
}

func getPgBouncerStats(stats statsd.Statter, pg bouncer) {
	db, err := sql.Open("postgres", pg.dsn)
	if err != nil {
		log.Println(err)
		return
	}
	defer closeResource(db)

	rows, err := db.Query("show pools")
	if err != nil {
		log.Println(err)
		return
	}
	defer closeResource(rows)

	//fmt.Println("running through stats")
	err = iterateRows(stats, pg.port, rows)
	if err != nil {
		log.Println(err)
	}
}

func logPgBouncerStats(pgs []bouncer) {
	// XXX AGB: this maintains 2 open connection, 1 to pgbouncer and 1
	// to graphite. I do not think this will be a problem because of
	// the serial nature of this process and the fact that pgbouncer can
	// maintain tens of thousands of connections. 2014-07-15
	var statsClient statsd.Statter
	var err error

	seconds := time.Duration(interval)
	if server == "" {
		statsClient, err = statsd.NewNoopClient()
	} else {
		statsClient, err = statsd.NewBufferedClient(server, "", seconds*time.Second, 0)
	}
	if err != nil {
		log.Println(err)
	}
	//this should be the connection to graphite?

	defer closeResource(statsClient)

	for _, pg := range pgs {
		//fmt.Println(pg.dsn)
		getPgBouncerStats(statsClient, pg)
	}
}

func pollPgBouncerStat(pgs []bouncer) {
	seconds := time.Duration(interval)
	for _ = range time.Tick(seconds * time.Second) {
		logPgBouncerStats(pgs)
	}
}

// Implementation of flag.Value interface.
func (p *intslice) String() string {
	return fmt.Sprintf("%d", *p)
}

func (p *intslice) Set(value string) error {
	tmp, err := strconv.Atoi(value)
	if err != nil {
		log.Fatal("Unable to parse port")
	}
	*p = append(*p, tmp)
	return nil
}

func init() {
	flag.StringVar(&cluster, "cluster", "", "cluster name to log under in statsd")
	flag.StringVar(&server, "server", "", "statsd server to send to")
	flag.IntVar(&interval, "interval", 10, "number of seconds between fetching stats.")

	flag.StringVar(&host, "host", "/var/run/postgresql", "host to connect to or domain socket location")
	flag.Var(&ports, "port", "pgbouncer port on the host. one port can be specified per flag.")
	flag.StringVar(&user, "user", "pgbouncer-stats", "pgbouncer admin user name")
	flag.StringVar(&pass, "pass", "", "password for connection")
	flag.StringVar(&schema, "schema", "pgbouncer", "database schema for the connection")
}

func main() {
	flag.Parse()
	if cluster == "" {
		log.Fatal("Must specify a cluster")
	}

	if len(ports) == 0 {
		log.Fatal("Must specify at least one port")
	}

	msg := fmt.Sprintf("Starting pgbouncer-stats: %v second interval on %v to %v", interval, cluster, server)
	log.Println(msg)

	var pgs []bouncer
	for _, port := range ports {
		pg := new(bouncer)
		if pass != "" {
			pg.dsn = fmt.Sprintf("sslmode=disable host=%v port=%v user=%v password=%v dbname='%v' binary_parameters=yes",
				host, port, user, pass, schema)
		} else {
			pg.dsn = fmt.Sprintf("sslmode=disable host=%v port=%v user=%v dbname='%v' binary_parameters=yes",
				host, port, user, schema)
		}
		pg.port = port
		pgs = append(pgs, *pg)
	}
	go pollPgBouncerStat(pgs)

	select {}
}

type closeable interface {
	Close() error
}

func closeResource(res closeable) {
	if res == nil {
		return
	}

	err := res.Close()
	if err != nil {
		log.Println(err)
	}
}
