/* Licensed to the Apache Software Foundation (ASF) under one or more
contributor license agreements.  See the NOTICE file distributed with
this work for additional information regarding copyright ownership.
The ASF licenses this file to You under the Apache License, Version 2.0
(the "License"); you may not use this file except in compliance with
the License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. */

package siesta

import "testing"

var emptyMetadataRequestBytes = []byte{0x00, 0x00, 0x00, 0x00}
var asdMetadataRequestBytes = []byte{0x00, 0x00, 0x00, 0x01, 0x00, 0x03, 0x61, 0x73, 0x64}
var multipleTopicsMetadataRequestBytes = []byte{0x00, 0x00, 0x00, 0x03, 0x00, 0x03, 0x61, 0x73, 0x64, 0x00, 0x03, 0x7a, 0x78, 0x63, 0x00, 0x03, 0x71, 0x77, 0x65}

var emptyMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}
var brokerMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x00, 0x00, 0x23, 0x84, 0x00, 0x00, 0x00, 0x00}
var topicMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x6c, 0x6f, 0x67, 0x73, 0x31, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00}

var invalidBrokersLengthMetadataResponseBytes = []byte{0x00, 0x00, 0x01}
var invalidMetadataLengthMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01}
var invalidBrokerNodeIDMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00}
var invalidBrokerHostMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f}
var invalidBrokerPortMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x00, 0x00, 0x23}
var invalidTopicMetadataErrorCodeMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00}
var invalidTopicMetadataTopicNameMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x6c, 0x6f, 0x67}
var invalidPartitionMetadataLengthMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x6c, 0x6f, 0x67, 0x73, 0x31, 0x00, 0x00, 0x00}
var invalidPartitionMetadataErrorCodeMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x6c, 0x6f, 0x67, 0x73, 0x31, 0x00, 0x00, 0x00, 0x02, 0x00}
var invalidPartitionMetadataPartitionMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x6c, 0x6f, 0x67, 0x73, 0x31, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00}
var invalidPartitionMetadataLeaderMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x6c, 0x6f, 0x67, 0x73, 0x31, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00}
var invalidPartitionMetadataReplicasLengthMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x6c, 0x6f, 0x67, 0x73, 0x31, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}
var invalidPartitionMetadataReplicaMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x6c, 0x6f, 0x67, 0x73, 0x31, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00}
var invalidPartitionMetadataIsrLengthMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x6c, 0x6f, 0x67, 0x73, 0x31, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}
var invalidPartitionMetadataIsrMetadataResponseBytes = []byte{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x6c, 0x6f, 0x67, 0x73, 0x31, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00}

func TestTopicMetadataRequest(t *testing.T) {
	emptyMetadataRequest := new(TopicMetadataRequest)
	testRequest(t, emptyMetadataRequest, emptyMetadataRequestBytes)

	asdMetadataRequest := NewMetadataRequest([]string{"asd"})
	testRequest(t, asdMetadataRequest, asdMetadataRequestBytes)

	multipleTopicsMetadataRequest := NewMetadataRequest([]string{"asd", "zxc", "qwe"})
	testRequest(t, multipleTopicsMetadataRequest, multipleTopicsMetadataRequestBytes)
}

func TestTopicMetadataResponse(t *testing.T) {
	emptyMetadataResponse := new(MetadataResponse)
	decode(t, emptyMetadataResponse, emptyMetadataResponseBytes)
	assertFatal(t, len(emptyMetadataResponse.Brokers), 0)
	assertFatal(t, len(emptyMetadataResponse.TopicsMetadata), 0)

	brokerMetadataResponse := new(MetadataResponse)
	decode(t, brokerMetadataResponse, brokerMetadataResponseBytes)
	assertFatal(t, len(brokerMetadataResponse.Brokers), 1)
	broker := brokerMetadataResponse.Brokers[0]
	assert(t, broker.ID, int32(0))
	assert(t, broker.Host, "localhost")
	assert(t, broker.Port, int32(9092))
	assertFatal(t, len(brokerMetadataResponse.TopicsMetadata), 0)

	topicMetadataResponse := new(MetadataResponse)
	decode(t, topicMetadataResponse, topicMetadataResponseBytes)
	assertFatal(t, len(topicMetadataResponse.Brokers), 0)
	assertFatal(t, len(topicMetadataResponse.TopicsMetadata), 1)
	meta := topicMetadataResponse.TopicsMetadata[0]
	assert(t, meta.Topic, "logs1")
	assert(t, meta.Error, ErrNoError)
	assertFatal(t, len(meta.PartitionsMetadata), 2)
	partition0 := meta.PartitionsMetadata[1]
	assert(t, partition0.PartitionID, int32(0))
	assert(t, partition0.Error, ErrNoError)
	assert(t, partition0.ISR, []int32{0})
	assert(t, partition0.Leader, int32(0))
	assert(t, partition0.Replicas, []int32{0})

	partition1 := meta.PartitionsMetadata[0]
	assert(t, partition1.PartitionID, int32(1))
	assert(t, partition1.Error, ErrNoError)
	assert(t, partition1.ISR, []int32{0})
	assert(t, partition1.Leader, int32(0))
	assert(t, partition1.Replicas, []int32{0})

	decodeErr(t, new(MetadataResponse), invalidBrokersLengthMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidBrokersLength))
	decodeErr(t, new(MetadataResponse), invalidMetadataLengthMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidMetadataLength))
	decodeErr(t, new(MetadataResponse), invalidBrokerNodeIDMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidBrokerNodeID))
	decodeErr(t, new(MetadataResponse), invalidBrokerHostMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidBrokerHost))
	decodeErr(t, new(MetadataResponse), invalidBrokerPortMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidBrokerPort))
	decodeErr(t, new(MetadataResponse), invalidTopicMetadataErrorCodeMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidTopicMetadataErrorCode))
	decodeErr(t, new(MetadataResponse), invalidTopicMetadataTopicNameMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidTopicMetadataTopicName))
	decodeErr(t, new(MetadataResponse), invalidPartitionMetadataLengthMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidPartitionMetadataLength))
	decodeErr(t, new(MetadataResponse), invalidPartitionMetadataErrorCodeMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidPartitionMetadataErrorCode))
	decodeErr(t, new(MetadataResponse), invalidPartitionMetadataPartitionMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidPartitionMetadataPartition))
	decodeErr(t, new(MetadataResponse), invalidPartitionMetadataLeaderMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidPartitionMetadataLeader))
	decodeErr(t, new(MetadataResponse), invalidPartitionMetadataReplicasLengthMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidPartitionMetadataReplicasLength))
	decodeErr(t, new(MetadataResponse), invalidPartitionMetadataReplicaMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidPartitionMetadataReplica))
	decodeErr(t, new(MetadataResponse), invalidPartitionMetadataIsrLengthMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidPartitionMetadataIsrLength))
	decodeErr(t, new(MetadataResponse), invalidPartitionMetadataIsrMetadataResponseBytes, NewDecodingError(ErrEOF, reasonInvalidPartitionMetadataIsr))
}
