variable "pg_cluster" {
  description = "Things like 'prod', 'replay', etc to uniquely identify the postgres cluster of master and replicas"
}
variable "r53_zone_id" {
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "security_groups" {
  description = "Comma separated list of security group ids"
}
variable "subnet_id" {
  description = "Use the subnet to sprinkle hosts among availability zones"
}
variable "twitch_environment" {
  description = "IGNORED -- here to remind us that we want this in the new VPC"
}

variable "ami" {
  default = "ami-1d1e0f2d"
}
variable "cluster" {
  default = "discovery-postgres"
  description = "The puppet hiera cluster for configuration"
}
variable "env" {
  default = "dev"
  description = "prod or dev -- used for naming"
}
variable "instance_name_prefix" {
  default = "discovery-postgres"
  description = "The dns name prefix to use -- usually the same as var.cluster"
}
variable "instance_type" {
  default = "c4.large"
}
variable "owner_tag" {
  default = "Core Data <core-data@twitch.tv>"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "project_tag" {
  default = "web/discovery"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "root_volume_size" {
  default = "250"
  description = "Root volume size in GB"
}

resource "aws_instance" "instance" {
  ami = "${var.ami}"
  instance_type = "${var.instance_type}"
  security_groups = ["${split(",", var.security_groups)}"]
  subnet_id = "${var.subnet_id}"

  user_data = <<END_OF_STRING
#cloud-config
preserve_hostname: true
manage_etc_hosts: false
bootcmd:
 - cloud-init-per instance my_set_hostname sh -xc "echo '${var.instance_name_prefix}-$INSTANCE_ID' | sed -e 's/-i-/-/g' > /etc/hostname; hostname -F /etc/hostname"
 - cloud-init-per instance my_etc_hosts sh -xc "sed -i -e '/^127.0.1.1/d' /etc/hosts; echo 127.0.1.1 $(cat /etc/hostname).${var.env}.us-west2.justin.tv $(cat /etc/hostname) >> /etc/hosts"
runcmd:
 - echo "cluster=${var.cluster}" > /etc/facter/facts.d/cluster.txt
 - echo "pg_cluster=${var.cluster}.${var.pg_cluster}" > /etc/facter/facts.d/pg_cluster.txt
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - apt-get update
 - apt-get upgrade
 - puppet agent --test --server=puppet.internal.justin.tv
END_OF_STRING

  root_block_device {
    volume_size = "${var.root_volume_size}"
    volume_type = "gp2"
  }
  tags {
    Name = "${var.cluster}/${var.pg_cluster}"
    Owner = "${var.owner_tag}"
    Project = "${var.project_tag}"
  }
}

resource "aws_route53_record" "dns" {
  name = "${var.instance_name_prefix}-${replace(aws_instance.instance.id, "i-", "")}.${var.env}.us-west2.justin.tv"
  records = ["${aws_instance.instance.private_ip}"]
  ttl = "300"
  type = "A"
  zone_id = "${var.r53_zone_id}"
}
