variable "twitch_dns_access_key" {}
variable "twitch_dns_secret_key" {}
provider "aws" {
  access_key = "${var.twitch_dns_access_key}"
  alias = "twitch_dns"
  region = "us-west-2"
  secret_key = "${var.twitch_dns_secret_key}"
}
variable "twitch_aws_access_key" {}
variable "twitch_aws_secret_key" {}
provider "aws" {
  access_key = "${var.twitch_aws_access_key}"
  alias = "twitch_web_aws"
  region = "us-west-2"
  secret_key = "${var.twitch_aws_secret_key}"
}

variable "aws_account" {
  description = "twitch account to use"
}
variable "cluster" {
  description = "The puppet hiera cluster for configuration"
}
variable "pg_cluster" {
  description = "Things like 'prod', 'replay', etc to uniquely identify the postgres cluster of master and replicas"
}
variable "r53_zone_id" {
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "vpc_security_group_ids" {
  description = "Comma separated list of security group ids"
}
variable "subnet_id" {
  description = "Use the subnet to sprinkle hosts among availability zones"
}
variable "twitch_environment" {
  description = "Used as the fact for development, production, and staging"
}

variable "ami" {
  default = "ami-5e26c36d"
}
variable "instance_name_prefix" {
  default = "postgres"
  description = "The dns name prefix to use -- usually the same as var.cluster"
}
variable "instance_type" {
  default = "c4.large"
}
variable "owner_tag" {
  default = "d8a <d8a@twitch.tv>"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "project_tag" {
  default = "web/postgresql"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "root_volume_size" {
  default = "250"
  description = "Root volume size in GB"
}
variable "instance_role" {
  default = ""
  description = "wale default role (needs to be changed)"
}
variable "cloudwatch_sns_topic_arn" {
  default = ""
  description = "cloudwatch topic for alarms to send notifications to"
}
variable "cloudwatch_monitoring" {
  default = ""
  description = "enable cloudwatch detailed monitoring"
}
variable "key_name"{
  default = ""
}
variable "associate_public_ip_address"{
  default = "false"
}

resource "aws_instance" "instance" {
  ami = "${var.ami}"
  instance_type = "${var.instance_type}"
  vpc_security_group_ids = ["${split(",", var.vpc_security_group_ids)}"]
  subnet_id = "${var.subnet_id}"
  iam_instance_profile = "${var.instance_role}"
  monitoring = "${var.cloudwatch_monitoring}"
  key_name = "${var.key_name}"
  associate_public_ip_address = "${var.associate_public_ip_address}"

  user_data = <<END_OF_STRING
#cloud-config
preserve_hostname: true
manage_etc_hosts: false
bootcmd:
 - cloud-init-per instance my_set_hostname sh -xc "echo '${var.instance_name_prefix}-$INSTANCE_ID' | sed -e 's/-i-/-/g' > /etc/hostname; hostname -F /etc/hostname"
 - cloud-init-per instance my_etc_hosts sh -xc "sed -i -e '/^127.0.1.1/d' /etc/hosts; echo 127.0.1.1 $(cat /etc/hostname).${var.twitch_environment}.${var.aws_account}.us-west2.justin.tv $(cat /etc/hostname) >> /etc/hosts"
runcmd:
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - echo "cluster=${var.cluster}" > /etc/facter/facts.d/cluster.txt
 - echo "pg_cluster=${var.cluster}.${var.pg_cluster}" > /etc/facter/facts.d/pg_cluster.txt
 - echo "twitch_environment=${var.twitch_environment}" > /etc/facter/facts.d/twitch_environment.txt
 - apt-get update
 - apt-get upgrade
 - puppet agent --test --server=puppet.internal.justin.tv
END_OF_STRING

  root_block_device {
    volume_size = "${var.root_volume_size}"
    volume_type = "gp2"
  }
  tags {
    Name = "${var.cluster}/${var.pg_cluster}"
    Owner = "${var.owner_tag}"
    Project = "${var.project_tag}"
  }
}

resource "aws_route53_record" "dns" {
  name = "${var.instance_name_prefix}-${replace(aws_instance.instance.id, "i-", "")}.${var.twitch_environment}.${var.aws_account}.us-west2.justin.tv"
  provider = "aws.twitch_dns"
  records = ["${aws_instance.instance.private_ip}"]
  ttl = "300"
  type = "A"
  zone_id = "${var.r53_zone_id}"
}

output "aws_instance_id" {
  value = "${aws_instance.instance.id}"
}
