variable "twitch_aws_access_key" {}
variable "twitch_aws_secret_key" {}

provider "aws" {
  access_key = "${var.twitch_aws_access_key}"
  region = "us-west-2"
  secret_key = "${var.twitch_aws_secret_key}"
}

module "base" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/release/terraform.git//base"
}

module "common" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/web/terraform.git//modules/variables/twitch-web-dev"
}
 
variable "aws_account"{
  default = "web-dev"
}

variable "min_size"{
  default = 0
}

variable "max_size"{
  default = 0
}

variable "desired_capacity"{
  default = 0
}


variable "lambda_arn" {
  #default = "arn:aws:lambda:us-west-2:465369119046:function:asg_route53_update"
  description = "add the arn of asg"
}

variable "aws_profile" {
  default = "postgres"
}

variable "twitch_environment"{
  default = "development"
} 

variable "ami" {
  default = "ami-5e26c36d"
}

variable "cluster" {
  #default = "rails-postgres"
  description = "The puppet hiera cluster for configuration"
}

variable "pg_cluster"{
  #default = "zookeeper"
  description = "Things like 'prod', 'replay', etc to uniquely identify the postgres cluster of master and replicas"
}

variable "env" {
  default = "dev"
  description = "prod or dev -- used for naming"
}
variable "instance_name_prefix" {
  #default = "rails-postgres-aws"
  description = "The dns name prefix to use -- usually the same as var.cluster"
}
variable "instance_type" {
  default = "i2.8xlarge"
}

variable "owner_tag" {
  default = "d8a <d8a@justin.tv>"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "project_tag" {
  default = "web/rails-postgres"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "root_volume_size" {
  default = "250"
  description = "Root volume size in GB"
}
variable "subnet_id" {
  #default = "subnet-b1af88d4,subnet-1d76336a,subnet-cbd54a92"
} 
variable "security_groups" {
  description = "Comma separated list of security group ids"
  #default ="sg-3cd6a958"
}
variable "enable_monitoring"{
  description = "enable detailed monitoring"
}
variable "instance_role"{
    description = "instance role to launch instance with"
}
variable "load_balancers"{
  description = "loadbalancer to attach to asg"
}

resource "aws_launch_configuration" "raid_as_conf" {
  image_id = "${var.ami}"
  associate_public_ip_address = "false"
  instance_type = "${var.instance_type}"
  security_groups = ["${split(",", var.security_groups)}"]
  enable_monitoring = "${var.enable_monitoring}"
  iam_instance_profile = "${var.instance_role}"

  user_data = <<END_OF_STRING
#cloud-config
manage_etc_hosts: false
preserve_hostname: true
bootcmd:
 - cloud-init-per instance my_set_hostname sh -xc "echo '${var.instance_name_prefix}-$INSTANCE_ID' | sed -e 's/-i-/-/g' > /etc/hostname; hostname -F /etc/hostname"
 - cloud-init-per instance my_etc_hosts sh -xc "sed -i -e '/^127.0.1.1/d' /etc/hosts; echo 127.0.1.1 $(cat /etc/hostname).${var.aws_account}.us-west2.justin.tv $(cat /etc/hostname) >> /etc/hosts"
runcmd:
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - echo "cluster=${var.cluster}" > /etc/facter/facts.d/cluster.txt
 - echo "pg_cluster=${var.cluster}.${var.pg_cluster}" > /etc/facter/facts.d/pg_cluster.txt
 - echo "twitch_environment=${var.twitch_environment}" > /etc/facter/facts.d/twitch_environment.txt
 - apt-get update
 - apt-get upgrade
 - apt-get install -y mdadm xfsprogs postgresql-common
 - mdadm --create --verbose /dev/md0 --level=0 --name=postgres --raid-devices=6 /dev/xvdf /dev/xvdg /dev/xvdh /dev/xvdi /dev/xvdj /dev/xvdk
 - mkfs.xfs -L postgres /dev/md0
 - mkdir /mnt/media
 - mount /dev/md0 /mnt/media
 - mv /var/lib/postgresql /mnt/media
 - ln -s /mnt/media/postgresql /var/lib/postgresql
 - puppet agent --test --server=puppet.internal.justin.tv
END_OF_STRING

  # These block devices are referenced in the prepare-sitedb script
  ephemeral_block_device {
    device_name = "/dev/xvdf"
    virtual_name = "ephemeral0"
  }
  ephemeral_block_device {
     device_name = "/dev/xvdg"
     virtual_name = "ephemeral1"
  }
  ephemeral_block_device {
    device_name = "/dev/xvdh"
    virtual_name = "ephemeral2"
  }
  ephemeral_block_device {
    device_name = "/dev/xvdi"
    virtual_name = "ephemeral3"
  }
  ephemeral_block_device {
    device_name = "/dev/xvdj"
    virtual_name = "ephemeral4"
  }
  ephemeral_block_device {
    device_name = "/dev/xvdk"
    virtual_name = "ephemeral5"
  }
  # ephemeral_block_device {
  #   device_name = "/dev/xvdl"
  #   virtual_name = "ephemeral6"
  # }
  # ephemeral_block_device {
  #   device_name = "/dev/xvdm"
  #   virtual_name = "ephemeral7"
  # }
  root_block_device {
    volume_size = "500"
    volume_type = "gp2"
  }
}




resource "aws_autoscaling_group" "asg" {
    name = "${var.cluster}_nonraid_${var.twitch_environment}"
    launch_configuration = "${aws_launch_configuration.raid_as_conf.name}"
    max_size = "${var.max_size}"
    min_size = "${var.min_size}"
    desired_capacity = "${var.desired_capacity}"
    load_balancers = ["${var.load_balancers}"]

    vpc_zone_identifier = ["${split(",", var.subnet_id)}"]

    lifecycle {
      create_before_destroy = true
    }

    tag {
      key = "Name"
      value = "${var.cluster}"
      propagate_at_launch = true
    }
    tag {
      key = "Owner"
      value = "${var.owner_tag}"
      propagate_at_launch = true
    }
    tag {
      key = "Project"
      value = "${var.project_tag}"
      propagate_at_launch = true
    }
}

output "asg_name"{  
  value = "${aws_autoscaling_group.asg.id}"
}