variable "twitch_dns_access_key" {}
variable "twitch_dns_secret_key" {}
provider "aws" {
  access_key = "${var.twitch_dns_access_key}"
  alias = "twitch_dns"
  region = "us-west-2"
  secret_key = "${var.twitch_dns_secret_key}"
}
variable "twitch_aws_access_key" {}
variable "twitch_aws_secret_key" {}
provider "aws" {
  access_key = "${var.twitch_aws_access_key}"
  #alias = "twitch_web_aws"
  region = "us-west-2"
  secret_key = "${var.twitch_aws_secret_key}"
}

variable "ami" {
}
variable "upstream_master" {
  description = "The source host for data provisioning"
}
variable "pg_cluster" {
  description = "Things like 'prod', 'replay', etc to uniquely identify the postgres cluster of master and replicas"
}
variable "r53_zone_id" {
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "vpc_security_group_ids" {
  description = "Comma separated list of security group ids"
}
variable "subnet_id" {
  description = "Use the subnet to sprinkle hosts among availability zones"
}
variable "twitch_environment" {
  description = "Used as the fact for development, production, and staging"
}

variable "aws_account" {
  description = "twitch account to use"
  default = "twitch-web-aws"
}
variable "cluster" {
  default = "rails-postgres"
  description = "The puppet hiera cluster for configuration"
}
variable "hostname_prefix" {
  default = "rails-postgres"
  description = "The dns name prefix to use -- usually the same as var.cluster"
}
variable "instance_role" {
  default = "postgres"
  description = "iam role for the node in ec2"
}
variable "instance_type" {
  default = "i2.8xlarge"
}
variable "owner_tag" {
  default = "d8a <d8a@justin.tv>"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "pg_role" {
  description = "Role of the node in the cluster, eg 'hot-spare'"
  default = "read-replica"
}
variable "project_tag" {
  default = "web/rails-postgres"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
resource "aws_instance" "instance" {
  ami = "${var.ami}"
  associate_public_ip_address = "false"
  iam_instance_profile = "${var.instance_role}"
  instance_type = "${var.instance_type}"
  vpc_security_group_ids = ["${split(",", var.vpc_security_group_ids)}"]
  subnet_id = "${var.subnet_id}"

  user_data = <<END_OF_STRING
#cloud-config
manage_etc_hosts: false
preserve_hostname: true
bootcmd:
 - cloud-init-per instance my_set_hostname sh -xc "echo '${var.hostname_prefix}-$INSTANCE_ID' | sed -e 's/-i-/-/g' > /etc/hostname; hostname -F /etc/hostname"
 - cloud-init-per instance my_etc_hosts sh -xc "sed -i -e '/^127.0.1.1/d' /etc/hosts; echo 127.0.1.1 $(cat /etc/hostname).${var.twitch_environment}.${var.aws_account}.us-west2.justin.tv $(cat /etc/hostname) >> /etc/hosts"
output: {all: '| tee -a /var/log/cloud-init-output.log'}
runcmd:
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - echo "cluster=${var.cluster}" > /etc/facter/facts.d/cluster.txt
 - echo "pg_cluster=${var.cluster}.${var.pg_cluster}" > /etc/facter/facts.d/pg_cluster.txt
 - echo "pg_role=${var.cluster}/${var.pg_cluster}/${var.pg_role}" > /etc/facter/facts.d/pg_role.txt
 - echo "twitch_environment=${var.twitch_environment}" > /etc/facter/facts.d/twitch_environment.txt
 - apt-get update
 - apt-get upgrade
 - apt-get install -y mdadm xfsprogs postgresql-common
 - mdadm --create --verbose /dev/md0 --level=10 --name=postgres --raid-devices=4 /dev/xvdf /dev/xvdg /dev/xvdh /dev/xvdi
 - mkfs.xfs -L postgres /dev/md0
 - mkdir /mnt/media
 - mount /dev/md0 /mnt/media
 - mv /var/lib/postgresql /mnt/media
 - ln -s /mnt/media/postgresql /var/lib/postgresql
 - sed -i '$ d' /etc/fstab # Cut the /dev/xvdf line from fstab so that it won't stop the next boot
 - echo "UUID=$(blkid | grep '/dev/md0' | egrep -o '[a-z0-9]{8}\-[a-z0-9]{4}\-[a-z0-9]{4}\-[a-z0-9]{4}\-[a-z0-9]{12}') /mnt/media xfs noatime,nobarrier,logbufs=8,logbsize=256k 0 1" >> /etc/fstab
 - echo "$(date +'%F %T %Z') -- starting initial puppet run"
 - puppet agent --test --server=puppet.internal.justin.tv
 - echo "$(date +'%F %T %Z') -- starting second puppet run"
 - puppet agent --test # run a second time in case first run fails
 - service postgresql stop
 - echo "$(date +'%F %T %Z') -- starting data provisioning"
 - sudo -u postgres rm -rf /var/lib/postgresql/9.3/main
 - sudo -u postgres mkdir /var/lib/postgresql/9.3/main
 - sudo -u postgres chmod 0700 /var/lib/postgresql/9.3/main
 - sudo -u postgres /usr/lib/postgresql/9.3/bin/pg_basebackup -X stream -h "${var.upstream_master}" -D /var/lib/postgresql/9.3/main -U replication -c fast -R
 - echo "$(date +'%F %T %Z') -- database copied, finalizing puppet"
 - puppet agent --test
 - echo "$(date +'%F %T %Z') -- rejoining cluster"
 - service postgresql restart
END_OF_STRING

  # These block devices are referenced in the prepare-sitedb script
  ephemeral_block_device {
    device_name = "/dev/xvdf"
    virtual_name = "ephemeral0"
  }
  ephemeral_block_device {
     device_name = "/dev/xvdg"
     virtual_name = "ephemeral1"
  }
  ephemeral_block_device {
    device_name = "/dev/xvdh"
    virtual_name = "ephemeral2"
  }
  ephemeral_block_device {
    device_name = "/dev/xvdi"
    virtual_name = "ephemeral3"
  }
  root_block_device {
    volume_size = "500"
    volume_type = "gp2"
  }
  tags {
    Name = "${var.cluster}/${var.pg_cluster}/${var.pg_role}"
    Owner = "${var.owner_tag}"
    Project = "${var.project_tag}"
  }
}

resource "aws_route53_record" "dns" {
  name = "${var.hostname_prefix}-${replace(aws_instance.instance.id, "i-", "")}.${var.twitch_environment}.${var.aws_account}.us-west2.justin.tv"
  provider = "aws.twitch_dns"
  records = ["${aws_instance.instance.private_ip}"]
  ttl = "300"
  type = "A"
  zone_id = "${var.r53_zone_id}"
}

output "instance_id" {
  value = "${aws_instance.instance.id}"
}
