variable "ami" {
}
variable "upstream_master" {
  description = "The source host for data provisioning"
}
variable "pg_cluster" {
  description = "Things like 'prod', 'replay', etc to uniquely identify the postgres cluster of master and replicas"
}
variable "r53_zone_id" {
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "vpc_security_group_ids" {
  description = "Comma separated list of security group ids"
}
variable "subnet_id" {
  description = "Use the subnet to sprinkle hosts among availability zones"
}
variable "twitch_environment" {
  description = "Used as the fact for development, production, and staging"
}
variable "puppet_environment" {
  description = "Branch of systems/puppet used for configuration."
}

variable "aws_account" {
  description = "twitch account to use"
  default = "twitch-web-aws"
}
variable "cluster" {
  default = "sitedb-postgres"
  description = "The puppet hiera cluster for configuration"
}
variable "hostname_prefix" {
  default = "sitedb-postgres"
  description = "The dns name prefix to use -- usually the same as var.cluster"
}
variable "instance_role" {
  default = "postgres"
  description = "iam role for the node in ec2"
}
variable "instance_type" {
  default = "i3.8xlarge"
}
variable "owner_tag" {
  default = "d8a <d8a@justin.tv>"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "pg_role" {
  description = "Role of the node in the cluster, eg 'hot-spare'"
  default = "read-replica"
}
variable "pg_version" {
  description = "The version of PostgreSQL to run"
  default = "9.3"
}
variable "project_tag" {
  default = "web/sitedb-postgres"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}

data "template_file" "master_20181022_ud" {
  template="${file("${path.module}/../templates.20170314/sitedb.20181022.user_data")}"
  vars {
    aws_account="${var.aws_account}"
    cluster="${var.cluster}"
    hostname_prefix="${var.hostname_prefix}"
    pg_cluster="${var.pg_cluster}"
    pg_role="${var.pg_role}"
    version="${var.pg_version}"
    twitch_environment="${var.twitch_environment}"
    branch="${var.puppet_environment}"
    upstream_master="${var.upstream_master}"
  }
}


resource "aws_instance" "instance" {
  ami = "${var.ami}"
  associate_public_ip_address = "false"
  iam_instance_profile = "${var.instance_role}"
  instance_type = "${var.instance_type}"
  subnet_id = "${var.subnet_id}"
  user_data = "${data.template_file.master_20181022_ud.rendered}"
  vpc_security_group_ids = ["${split(",", var.vpc_security_group_ids)}"]

/*
  ephemeral_block_device {
    device_name = "/dev/nvme0"
    virtual_name = "ephemeral0"
  }
  ephemeral_block_device {
     device_name = "/dev//dev/nvme0n1"
     virtual_name = "ephemeral1"
  }
  ephemeral_block_device {
    device_name = "/dev/nvme1"
    virtual_name = "ephemeral2"
  }
  ephemeral_block_device {
    device_name = "/dev/nvme1n1"
    virtual_name = "ephemeral3"
  }
  ephemeral_block_device {
    device_name = "/dev/nvme2"
    virtual_name = "ephemeral4"
  }
  ephemeral_block_device {
    device_name = "/dev/nvme2n1"
    virtual_name = "ephemeral5"
  }
  ephemeral_block_device {
    device_name = "/dev/nvme3"
    virtual_name = "ephemeral6"
  }
  ephemeral_block_device {
    device_name = "/dev/nvme3n1"
    virtual_name = "ephemeral7"
  }
  root_block_device {
    volume_size = "500"
    volume_type = "gp2"
  }
*/
  tags {
    Name = "${var.cluster}/${var.pg_cluster}/${var.pg_role}"
    Owner = "${var.owner_tag}"
    Project = "${var.project_tag}"
  }
  timeouts {
    create = "1h"
    delete = "1h"
  }
}

resource "aws_route53_record" "dns" {
  name = "${var.hostname_prefix}-${replace(aws_instance.instance.id, "i-", "")}.${var.twitch_environment}.${var.aws_account}.us-west2.justin.tv"
  provider = "aws.dns"
  records = ["${aws_instance.instance.private_ip}"]
  ttl = "300"
  type = "A"
  zone_id = "${var.r53_zone_id}"
}

output "instance_id" {
  value = "${aws_instance.instance.id}"
}

output "fqdn" {
  value = "${aws_route53_record.dns.name}"
}
