resource "aws_iam_instance_profile" "app" {
    name = "iam_profile_rds_buddy_${var.team}_${var.twitch_environment}"
    role = "${aws_iam_role.app.name}"
}

resource "aws_iam_role" "app" {
    name = "iam_role_rds_buddy_${var.team}_${var.twitch_environment}"
    assume_role_policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "sts:AssumeRole",
            "Principal": {
                "Service": "ec2.amazonaws.com"
            },
            "Effect": "Allow",
            "Sid": ""
        }
    ]
}
EOF
}

resource "aws_iam_policy" "rds_modify_instance_policy" {
    name = "rds-buddy-${var.team}-modify-instance-policy"
    policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "rds:ModifyDBInstance",
            "Effect": "Allow",
            "Resource": "*",
            "Condition": {
                "Null": {
                    "rds:db-tag/rds-buddy-${var.team}-managed": false
                }
            }
        },
        {
            "Action": "rds:ModifyDBInstance",
            "Effect": "Allow",
            "Resource": "*",
            "Condition": {
                "Null": {
                    "rds:db-tag/rds-buddy-${var.team}-store": false
                }
            }
        },
        {
            "Action": "rds:CreateDBInstance",
            "Effect": "Allow",
            "Resource": "*"
        }
    ]
}
EOF
}

resource "aws_iam_policy_attachment" "rds_modify_instance_attach" {
    name = "rds-buddy-${var.team}-modify-instance-attach"
    roles = ["${aws_iam_role.app.name}"]
    policy_arn = "${aws_iam_policy.rds_modify_instance_policy.arn}"
}

resource "aws_iam_policy" "rds_modify_cluster_policy" {
    name = "rds-buddy-${var.team}-modify-cluster-policy"
    policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "rds:ModifyDBCluster",
            "Effect": "Allow",
            "Resource": "*",
            "Condition": {
                "Null": {
                    "rds:cluster-tag/rds-buddy-${var.team}-managed": false
                }
            }
        }
    ]
}
EOF
}

resource "aws_iam_policy_attachment" "rds_modify_cluster_attach" {
    name = "rds-buddy-${var.team}-modify-cluster-attach"
    roles = ["${aws_iam_role.app.name}"]
    policy_arn = "${aws_iam_policy.rds_modify_cluster_policy.arn}"
}

resource "aws_iam_policy" "rds_buddy_policy" {
    name = "rds-buddy-${var.team}-access-policy"
    count = "${signum(var.raise_instance)}"
    policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "sts:AssumeRole",
            "Resource": [
                "${aws_iam_role.app.arn}",
                "${var.sandstorm_role_arn}",
                "arn:aws:iam::${var.science_aws_account_number}:role/zenyatta-${var.team}-putter-prod"
            ],
            "Effect": "Allow"
        },
        {
            "Action": ["rds:DescribeDB*", "rds:ListTagsForResource", "rds:DescribeDBSubnetGroups", "rds:CreateDBSubnetGroup", "rds:RestoreDB*", "rds:ModifyDB*", "rds:DeleteDB*"],
            "Resource": "*",
            "Effect": "Allow"
        },
        {
            "Action": ["cloudwatch:DeleteAlarms", "cloudwatch:DescribeAlarms", "cloudwatch:PutMetricAlarm"],
            "Resource": "*",
            "Effect": "Allow"
        },
        {
            "Action": "SNS:ListTopics",
            "Resource": "*",
            "Effect": "Allow"
        },
        {
            "Action": ["EC2:DescribeInstances", "EC2:CreateTags", "EC2:RunInstances", "EC2:DescribeSecurityGroups", "EC2:CreateKeyPair", "EC2:DescribeKeyPairs"],
            "Resource": "*",
            "Effect": "Allow"
        },
        {
          "Action": ["EC2:TerminateInstances"],
          "Resource": "*",
          "Effect": "Allow",
          "Condition": {
            "StringLike": {
              "ec2:ResourceTag/Name": "*zenyatta-worker"
            }
          }
        },
        {
            "Action": ["elasticache:DescribeCacheSubnetGroups", "elasticache:CreateCacheSubnetGroup", "elasticache:DescribeCacheClusters", "elasticache:CreateCacheCluster"],
            "Resource": "*",
            "Effect": "Allow"
        },
        {
            "Action": ["autoscaling:DescribeAutoScalingInstances", "autoscaling:DescribeAutoScalingGroups"],
            "Resource": "*",
            "Effect": "Allow"
        },
        {
            "Action": ["sts:DecodeAuthorizationMessage"],
            "Resource": "*",
            "Effect": "Allow"
        },
        {
            "Action": ["elasticmapreduce:ListClusters", "elasticmapreduce:RunJobFlow",
                       "elasticmapreduce:DescribeCluster", "elasticmapreduce:AddJobFlowSteps",
                       "elasticmapreduce:ListSteps"],
            "Resource": "*",
            "Effect": "Allow"
        },
        {
            "Action": ["SNS:Publish"],
            "Resource": ["arn:aws:sns:us-west-2:641044725657:zenyatta-finished"],
            "Effect": "Allow"
        },
        {
            "Action": "iam:PassRole",
            "Effect": "Allow",
            "Resource": [
                "${aws_iam_role.emr.arn}",
                "${aws_iam_role.app.arn}"
            ]
        },
        {
            "Action": [
                "s3:Get*",
                "s3:Put*",
                "s3:List*",
                "s3:Delete*"
            ],
            "Effect": "Allow",
            "Resource": [
                "arn:aws:s3:::${var.etl_bucket}*",
                "arn:aws:s3:::${var.etl_bucket}/*"
            ]
        }
    ]
}
EOF
}

resource "aws_iam_policy_attachment" "rds_buddy_attach" {
    name = "rds-buddy-${var.team}-access-attach"
    count = "${signum(var.raise_instance)}"
    roles = ["${aws_iam_role.app.name}"]
    policy_arn = "${aws_iam_policy.rds_buddy_policy.arn}"
}

resource "aws_iam_instance_profile" "instance-profile-emr-ec2" {
    name = "rds-buddy-emr-ec2-${var.team}-${var.twitch_environment}"
    role = "${aws_iam_role.emr-ec2.name}"
}

# IAM Role for EC2 Instance Profile
resource "aws_iam_role" "emr-ec2" {
    name = "rds-buddy-emr-ec2-${var.team}-${var.twitch_environment}"
    description = "Role to allow EC2 instances in an EMR cluster to access other AWS services such as S3."
    assume_role_policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "sts:AssumeRole",
            "Principal": {
                "Service": "ec2.amazonaws.com"
            },
            "Effect": "Allow",
            "Sid": ""
        }
    ]
}
EOF
}

# IAM role for EMR Service
resource "aws_iam_role" "emr" {
    name = "rds-buddy-emr-${var.team}-${var.twitch_environment}"
    description = "Role to allow EMR to access other AWS services such as EC2 on your behalf."
    assume_role_policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "sts:AssumeRole",
            "Principal": {
                "Service": "elasticmapreduce.amazonaws.com"
            },
            "Effect": "Allow",
            "Sid": ""
        }
    ]
}
EOF
}

resource "aws_iam_policy" "policy" {
    name = "rds-buddy-emr-ec2-${var.team}-access-policy"
    policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "ec2:*",
                "elasticmapreduce:*"
            ],
            "Resource": "*"
        },
        {
            "Effect": "Allow",
            "Action": "sts:AssumeRole",
            "Resource": [
                "arn:aws:iam::${var.science_aws_account_number}:role/zenyatta-${var.team}-putter-prod"
            ]
        }
    ]
}
EOF
}

resource "aws_iam_policy_attachment" "emr-ec2-role-policy-attach" {
    name = "rds-buddy-emr-ec2-${var.team}-policy-attach"
    roles = ["${aws_iam_role.emr-ec2.name}"]
    policy_arn = "${aws_iam_policy.policy.arn}"
}


resource "aws_iam_policy_attachment" "emr-ec2-amz-role-policy-attach" {
    name = "rds-buddy-emr-ec2-${var.team}-amz-policy-attach"
    roles = ["${aws_iam_role.emr-ec2.name}"]
    policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonElasticMapReduceforEC2Role"
}

resource "aws_iam_policy_attachment" "emr-amz-role-policy-attach" {
    name = "rds-buddy-emr-${var.team}-amz-policy-attach"
    roles = ["${aws_iam_role.emr.name}"]
    policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonElasticMapReduceRole"
}
