output "role_arn" {
    value = "${aws_iam_role.role.arn}"
}

# Set region
data "aws_region" "current" {
  current = true
}

# Get account ID
data "aws_caller_identity" "current" {}

# Create a bucket to store Cloudtrail logs

resource "aws_s3_bucket" "bucket" {
  bucket = "cloudtrail-events-${data.aws_caller_identity.current.account_id}"

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "AWSCloudTrailAclCheck",
            "Effect": "Allow",
            "Principal": {
              "Service": "cloudtrail.amazonaws.com"
            },
            "Action": "s3:GetBucketAcl",
            "Resource":
"arn:aws:s3:::cloudtrail-events-${data.aws_caller_identity.current.account_id}"
        },
        {
            "Sid": "AWSCloudTrailWrite",
            "Effect": "Allow",
            "Principal": {
              "Service": "cloudtrail.amazonaws.com"
            },
            "Action": "s3:PutObject",
            "Resource":
"arn:aws:s3:::cloudtrail-events-${data.aws_caller_identity.current.account_id}/*",
            "Condition": {
                "StringEquals": {
                    "s3:x-amz-acl": "bucket-owner-full-control"
                }
            }
        }
    ]
}
EOF
}

# Create our Cloudtrail Trail

resource "aws_cloudtrail" "trail" {
  name = "cloudtrail-events-${data.aws_caller_identity.current.account_id}"

  s3_bucket_name = "cloudtrail-events-${data.aws_caller_identity.current.account_id}"

  include_global_service_events = true
  is_multi_region_trail         = true
  enable_log_file_validation    = true
  depends_on                    = ["aws_s3_bucket.bucket"]
}

# Create the policy that can access Cloudtrail configs and the S3 bucket where
# the logs are stored

resource "aws_iam_policy" "policy" {
  name        = "Read_Only_CloudTrail_Policy"
  path        = "/"
  description = "Enables access to Cloudtrail logs"

  policy = <<EOF
{
"Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "cloudtrail:DescribeTrails",
                "cloudtrail:GetEventSelectors",
                "cloudtrail:GetTrailStatus",
                "cloudtrail:ListPublicKeys",
                "cloudtrail:ListTags",
                "cloudtrail:LookupEvents"
            ],
            "Resource": [
                "*"
            ]
        },
        {
            "Effect": "Allow",
            "Action": [
                "s3:Get*",
                "s3:List*",
                "s3:Head*"
            ],
            "Resource": [
                "arn:aws:s3:::cloudtrail-events*"
            ]
        }
    ]
}
EOF
}

# Create the role that security can assume

resource "aws_iam_role" "role" {
  name = "twitch-security-audit-limited"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": "arn:aws:iam::508423404353:root"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

# Apply the policy to the role

resource "aws_iam_role_policy_attachment" "pol-attach" {
  role       = "${aws_iam_role.role.name}"
  policy_arn = "${aws_iam_policy.policy.arn}"
}

resource "aws_iam_role_policy_attachment" "pol-attach-2" {
  role       = "${aws_iam_role.role.name}"
  policy_arn = "arn:aws:iam::aws:policy/SecurityAudit"
}

# Create a policy that allows Lambda to start logging and send SNS alerts
resource "aws_iam_policy" "lambda_policy" {
  name        = "lambda_cloudtrail_alert_policy"
  path        = "/"
  description = "Enables StartLogging and SNS Notifications for Lambda"

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "logs:CreateLogGroup",
                "logs:CreateLogStream",
                "logs:PutLogEvents"
                ],
            "Resource": "arn:aws:logs:*:*:*"
        },
        {
            "Effect": "Allow",
            "Action": [
                "sns:Publish"
            ],
            "Resource": "arn:aws:sns:us-west-2:508423404353:twitch-sirt"
        },
        {
            "Effect": "Allow",
            "Action": [
                "cloudtrail:StartLogging"
            ],
            "Resource": [
                "arn:aws:cloudtrail:*"
            ]
        }
    ]
}
EOF
}

# Create a role that allows lambda execution

resource "aws_iam_role" "lambda_role" {
  name = "lambda_execute_cloudtrail_role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "lambda.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

# Attach lambda policy to role
resource "aws_iam_role_policy_attachment" "lambda-attach" {
  role       = "${aws_iam_role.lambda_role.name}"
  policy_arn = "${aws_iam_policy.lambda_policy.arn}"
}



# Create the lambda

resource "aws_lambda_function" "lambda_function" {
  filename         = "${path.module}/lambda.zip"
  function_name    = "lambda_cloudtrail_monitor"
  role             = "${aws_iam_role.lambda_role.arn}"
  handler          = "lambda_cloudtrail_alert.lambda_handler"
  source_code_hash = "${base64sha256(file("${path.module}/lambda.zip"))}"
  runtime          = "python2.7"
  }

# Create a rule to fire the lambda
resource "aws_cloudwatch_event_rule" "console" {
  name        = "cloudwatch-event-guard"
  description = "Triggers on StopLogging, DeleteTrail, and UpdateTrail"

  event_pattern = <<PATTERN
{
  "detail":
    {
     "eventName": [ "StopLogging","DeleteTrail","UpdateTrail"],
     "eventSource":["cloudtrail.amazonaws.com"]
    },
  "detail-type":["AWS API Call via CloudTrail"],
  "source":["aws.cloudtrail"]
}
PATTERN
}

# define the target for the rule

resource "aws_cloudwatch_event_target" "lambda_target" {
  rule      = "${aws_cloudwatch_event_rule.console.name}"
  target_id = "1-cloudwatch-event-guard"
  arn       = "${aws_lambda_function.lambda_function.arn}"
}

# Add lambda invoke permission to rule
resource "aws_lambda_permission" "allow_cloudwatch" {
  statement_id   = "lambda_cloudtrail_monitor-${data.aws_caller_identity.current.account_id}"
  action         = "lambda:InvokeFunction"
  function_name  = "${aws_lambda_function.lambda_function.function_name}"
  principal      = "events.amazonaws.com"
  source_arn     = "arn:aws:events:${data.aws_region.current.name}:${data.aws_caller_identity.current.account_id}:rule/cloudwatch-event-guard"
}
