variable "twitch_aws_access_key" {}
variable "twitch_aws_secret_key" {}

provider "aws" {
  access_key = "${var.twitch_aws_access_key}"
  region = "us-west-2"
  secret_key = "${var.twitch_aws_secret_key}"
}

module "base" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/release/terraform.git//base?ref=terraform07"
}

module "common" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/web/terraform.git//modules/variables/twitch-web-dev"
}
 
variable "aws_account"{
  default = "web-dev"
}

variable "min_size"{
  default = 1
}

variable "max_size"{
  default = 1
}

variable "desired_capacity"{
  default = 1
}

variable "cluster_name"{
  default = "zookeeper_dev"  
}

variable "topic_name" {
  default = "zookeeper_dev_topic"
}


variable "lambda_arn" {
  default = "arn:aws:lambda:us-west-2:465369119046:function:asg_route53_update"
}

variable "aws_profile" {
  default = "zookeeper"
}

variable "twitch_environment"{
  default = "development"
} 

variable "ami" {
  default = "ami-5e26c36d"
}

variable "cluster" {
  default = "zk"
  description = "The puppet hiera cluster for configuration"
}

variable "pg_cluster"{
  default = "zookeeper"
}


variable "instance_role"{
  default = "d8a-zookeeper"
}

variable "env" {
  default = "dev"
  description = "prod or dev -- used for naming"
}
variable "instance_name_prefix" {
  default = "zk-aws"
  description = "The dns name prefix to use -- usually the same as var.cluster"
}
variable "instance_type" {
  default = "c4.large"
}
variable "name_tag" {
  default = "d8a/zookeeper"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "owner_tag" {
  default = "d8a <d8a@justin.tv>"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "project_tag" {
  default = "d8a/zookeeper"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "root_volume_size" {
  default = "250"
  description = "Root volume size in GB"
}
variable "subnet_id" {
  default = "subnet-b1af88d4,subnet-1d76336a,subnet-cbd54a92"
} 
variable "security_groups" {
  description = "Comma separated list of security group ids"
  default ="sg-3cd6a958"
}

variable "vpc" {
  default = "vpc-230f5e46"
}

variable "exhibitor_s3_config_file"{ 
}

variable "alert_interval" {
  default = "30000"
  description = "interval in milliseconds for checking status of zk and exhibitor"
}
variable "send_alert" {
  default = "false"
  description = "sending alert to pagerduty"
}
variable "statsd_cluster_name" {
  default = "zk_aws_web_dev"
  description = "the cluster name to appear in statsd/graphite"
}

variable "statsd_server" {
  default = "graphite-37c291f2.dev.us-west2.justin.tv:8125"
  description = "statsd/graphit server"
}

resource "aws_launch_configuration" "as_conf" {
    provider = "aws"
    image_id = "${var.ami}"
    instance_type = "${var.instance_type}"
    associate_public_ip_address = "false"
    lifecycle {
      create_before_destroy = true
    }
    security_groups = ["${split(",", var.security_groups)}"]
    iam_instance_profile = "${var.instance_role}"


    user_data = <<END_OF_STRING
#cloud-config
preserve_hostname: true
manage_etc_hosts: false
bootcmd:
 - cloud-init-per instance my_set_hostname sh -xc "echo '${var.instance_name_prefix}-$INSTANCE_ID' | sed -e 's/-i-/-/g' > /etc/hostname; hostname -F /etc/hostname"
 - cloud-init-per instance my_etc_hosts sh -xc "sed -i -e '/^127.0.1.1/d' /etc/hosts; echo 127.0.1.1 $(cat /etc/hostname).${var.env}.us-west2.justin.tv $(cat /etc/hostname) >> /etc/hosts"
runcmd:
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - echo "cluster=${var.cluster}" > /etc/facter/facts.d/cluster.txt
 - echo "twitch_environment=${var.twitch_environment}" > /etc/facter/facts.d/twitch_environment.txt
 - echo "vpc_id=${var.vpc}" > /etc/facter/facts.d/vpcid.txt
 - echo "exhibitor_s3_config_file=${var.exhibitor_s3_config_file}" > /etc/facter/facts.d/exhibitor_s3_config.txt
 - echo "domain_suffix=${var.env}.us-west2.justin.tv" > /etc/facter/facts.d/domain_suffix.txt
 - printf "statsdcluster=${var.statsd_cluster_name}\nstatsdserver=${var.statsd_server}" > /etc/facter/facts.d/zkalerts.txt
 - printf "alertinterval=${var.alert_interval}\nsendalert=${var.send_alert}" > /etc/facter/facts.d/zkstats.txt
 - apt-get update
 - apt-get upgrade
 - puppet agent --test --server=puppet.internal.justin.tv
END_OF_STRING

  root_block_device {
    volume_size = "${var.root_volume_size}"
    volume_type = "gp2"
  }

}

resource "aws_autoscaling_group" "asg" {
    name = "${var.cluster}_${var.twitch_environment}"
    launch_configuration = "${aws_launch_configuration.as_conf.name}"
    max_size = "${var.max_size}"
    min_size = "${var.min_size}"
    desired_capacity = "${var.desired_capacity}"

    vpc_zone_identifier = ["${split(",", var.subnet_id)}"]

    lifecycle {
      create_before_destroy = true
    }

    tag {
      key = "Name"
      value = "${var.cluster}"
      propagate_at_launch = true
    }
    tag {
      key = "Owner"
      value = "${var.owner_tag}"
      propagate_at_launch = true
    }
    tag {
      key = "Project"
      value = "${var.project_tag}"
      propagate_at_launch = true
    }
}

output "asg_name"{  
  value = "${aws_autoscaling_group.asg.id}"
}
