variable "r53_zone_id" {
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "security_groups" {
  description = "Comma separated list of security group ids"
}
variable "subnet_id" {
  description = "Use the subnet to sprinkle hosts among availability zones"
}
variable "twitch_environment" {
  description = "Used as the fact for development, production, and staging"
}
variable "vpc" {
  description = "VPC where this instance runs"
}
variable "exhibitor_s3_config_file" {
  description = "The location of exhibitor configuration file in S3"
}

variable "statsd_cluster_name" {
  description = "the cluster name to appear in statsd/graphite"
}

variable "statsd_server" {
  description = "statsd/graphit server"
}

variable "alert_interval" {
  description = "interval in milliseconds for checking status of zk and exhibitor"
}

variable "send_alert" {
  description = "sending alert to pagerduty"
}

variable "ami" {
  default = "ami-1d1e0f2d"
}
variable "cluster" {
  default = "zk"
  description = "The puppet hiera cluster for configuration"
}
variable "env" {
  default = "dev"
  description = "prod or dev -- used for naming"
}
variable "instance_name_prefix" {
  default = "zk"
  description = "The dns name prefix to use -- usually the same as var.cluster"
}
variable "instance_type" {
  default = "c4.large"
}
variable "name_tag" {
  default = "d8a/zookeeper"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "owner_tag" {
  default = "d8a <d8a@justin.tv>"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "project_tag" {
  default = "d8a/zookeeper"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}
variable "root_volume_size" {
  default = "250"
  description = "Root volume size in GB"
}
variable "source_tag" {
  default = "lib.20151006"
  description = "Tag used on resource tags and names to help locate resources in AWS"
}

resource "aws_instance" "instance" {
  ami = "${var.ami}"
  associate_public_ip_address = "false"
  iam_instance_profile = "d8a_zookeeper"
  instance_type = "${var.instance_type}"
  security_groups = ["${split(",", var.security_groups)}"]
  subnet_id = "${var.subnet_id}"

  user_data = <<END_OF_STRING
#cloud-config
preserve_hostname: true
manage_etc_hosts: false
bootcmd:
 - cloud-init-per instance my_set_hostname sh -xc "echo '${var.instance_name_prefix}-$INSTANCE_ID' | sed -e 's/-i-/-/g' > /etc/hostname; hostname -F /etc/hostname"
 - cloud-init-per instance my_etc_hosts sh -xc "sed -i -e '/^127.0.1.1/d' /etc/hosts; echo 127.0.1.1 $(cat /etc/hostname).${var.env}.us-west2.justin.tv $(cat /etc/hostname) >> /etc/hosts"
runcmd:
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - echo "cluster=${var.cluster}" > /etc/facter/facts.d/cluster.txt
 - echo "twitch_environment=${var.twitch_environment}" > /etc/facter/facts.d/twitch_environment.txt
 - echo "vpc_id=${var.vpc}" > /etc/facter/facts.d/vpcid.txt
 - echo "exhibitor_s3_config_file=${var.exhibitor_s3_config_file}" > /etc/facter/facts.d/exhibitor_s3_config.txt
 - echo "domain_suffix=${var.env}.us-west2.justin.tv" > /etc/facter/facts.d/domain_suffix.txt
 - printf "statsdcluster=${var.statsd_cluster_name}\nstatsdserver=${var.statsd_server}" > /etc/facter/facts.d/zkalerts.txt
 - printf "alertinterval=${var.alert_interval}\nsendalert=${var.send_alert}" > /etc/facter/facts.d/zkstats.txt
 - apt-get update
 - apt-get upgrade
 - puppet agent --test --server=puppet.internal.justin.tv
END_OF_STRING

  root_block_device {
    volume_size = "${var.root_volume_size}"
    volume_type = "gp2"
  }
  tags {
    Name = "${var.name_tag}"
    Owner = "${var.owner_tag}"
    Project = "${var.project_tag}"
    Source = "${var.source_tag}"
  }
}

resource "aws_route53_record" "dns" {
  name = "${var.instance_name_prefix}-${replace(aws_instance.instance.id, "i-", "")}.${var.env}.us-west2.justin.tv"
  records = ["${aws_instance.instance.private_ip}"]
  ttl = "300"
  type = "A"
  zone_id = "${var.r53_zone_id}"
}
