module "base" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/release/terraform.git//base?ref=terraform07"
}

module "common" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/web/terraform.git//modules/variables/twitch-web-aws/"
}

provider "aws" {
  region = "us-west-2"
}

variable "twitch_web_aws_access_key" {}
variable "twitch_web_aws_secret_key" {}
variable "twitch_aws_access_key" {}
variable "twitch_aws_secret_key" {}


variable "data_a_cidr" {
  default = "10.194.204.0/24"
}

variable "data_b_cidr" {
  default = "10.194.220.0/24"
}
variable "data_c_cidr" {
  default = "10.194.236.0/24"
}

variable twitch_subnets {
  default = "10.0.0.0/8,192.16.64.0/21,199.9.248.0/21,185.42.204.0/22,127.0.0.1/32,38.99.10.97/32,192.168.0.0/16,45.113.128.0/22,103.53.48.0/22,192.108.239.0/24,23.160.0.0/24"
}

# consider making this a module
resource "aws_security_group" "zk_sec" {
  description = "security rules for the production aws web zookeeper"
  name = "zk_production_security"
  vpc_id = "${module.common.vpc_id}"

  egress {
    from_port = 0
    to_port = 0
    protocol = -1
    cidr_blocks = ["0.0.0.0/0"]
  }

  ingress {
    from_port = 22
    to_port = 22
    protocol = "tcp"
    cidr_blocks = ["${split(",", var.twitch_subnets)}"]
  }

  # exhibitor port
  ingress {
    from_port = 8080
    to_port = 8080
    protocol = "tcp"
    cidr_blocks = ["${split(",", var.twitch_subnets)}"]
  }

  # consul check port
  ingress {
    from_port = 8000
    to_port = 8000
    protocol = "tcp"
    cidr_blocks = ["${split(",", var.twitch_subnets)}"]
  }

  # zk client connection port
  ingress {
    from_port = 2181
    to_port = 2181
    protocol = "tcp"
    cidr_blocks = ["${var.data_a_cidr}", "${var.data_b_cidr}", "${var.data_c_cidr}"]
  }

  # peer communication port
  ingress {
    from_port = 2888
    to_port = 2888
    protocol = "tcp"
    cidr_blocks = ["${var.data_a_cidr}", "${var.data_b_cidr}", "${var.data_c_cidr}"]
  }

  # leader election port
  ingress {
    from_port = 3888
    to_port = 3888
    protocol = "tcp"
    cidr_blocks = ["${var.data_a_cidr}", "${var.data_b_cidr}", "${var.data_c_cidr}"]
  }

  tags {
    Name = "zk-production/security"
    Owner = "d8a <d8a@justin.tv>"
    Project = "web/zk-cluster"
  }
}

variable ec2_instance_size {
    default = "c4.xlarge"
    description = "size of machine for the cluster"
}

variable "send_alert" {
  default = "true"
  description = "sending alert to pagerduty"
}

variable "env" {
  default = "prod"
  description = "prod or dev -- used for naming"
}

variable "statsd_cluster_name" {
  default = "zk_aws_prod"
  description = "the cluster name to appear in statsd/graphite"
}

variable "statsd_server" {
  default = "graphite.internal.justin.tv:8125"
  description = "statsd/graphit server"
}

variable "alert_interval" {
  default = "30000"
  description = "interval in milliseconds for checking status of zk and exhibitor"
}

variable "twitch_environment" {
  default = "production"
  description = "Used as the fact for development, production, and staging"
}

module "zk_0" {
  twitch_aws_access_key = "${var.twitch_aws_access_key}"
  twitch_aws_secret_key = "${var.twitch_aws_secret_key}"
  twitch_web_aws_access_key = "${var.twitch_web_aws_access_key}"
  twitch_web_aws_secret_key = "${var.twitch_web_aws_secret_key}"

  env = "${var.env}"
  instance_type = "${var.ec2_instance_size}"
  r53_zone_id = "${module.base.r53}"
  security_groups = "${aws_security_group.zk_sec.id}"
  source = "../lib/lib.20151013"
  subnet_id = "${module.common.data_a}"
  twitch_environment = "${var.twitch_environment}"
  vpc = "${module.common.vpc_id}"
  exhibitor_s3_config_file = "twitch-web-prod:config/exhibitor/${module.common.vpc_id}/exhibitor.properties"
  statsd_cluster_name = "${var.statsd_cluster_name}"
  statsd_server = "${var.statsd_server}"
  alert_interval ="${var.alert_interval}"
  send_alert = "${var.send_alert}"
}

module "zk_1" {
  twitch_aws_access_key = "${var.twitch_aws_access_key}"
  twitch_aws_secret_key = "${var.twitch_aws_secret_key}"
  twitch_web_aws_access_key = "${var.twitch_web_aws_access_key}"
  twitch_web_aws_secret_key = "${var.twitch_web_aws_secret_key}"

  env = "${var.env}"
  instance_type = "${var.ec2_instance_size}"
  r53_zone_id = "${module.base.r53}"
  security_groups = "${aws_security_group.zk_sec.id}"
  source = "../lib/lib.20151013"
  subnet_id = "${module.common.data_a}"
  twitch_environment = "${var.twitch_environment}"
  vpc = "${module.common.vpc_id}"
  exhibitor_s3_config_file = "twitch-web-prod:config/exhibitor/${module.common.vpc_id}/exhibitor.properties"
  statsd_cluster_name = "${var.statsd_cluster_name}"
  statsd_server = "${var.statsd_server}"
  alert_interval = "${var.alert_interval}"
  send_alert = "${var.send_alert}"
}

module "zk_2" {
  twitch_aws_access_key = "${var.twitch_aws_access_key}"
  twitch_aws_secret_key = "${var.twitch_aws_secret_key}"
  twitch_web_aws_access_key = "${var.twitch_web_aws_access_key}"
  twitch_web_aws_secret_key = "${var.twitch_web_aws_secret_key}"

  env = "${var.env}"
  instance_type = "${var.ec2_instance_size}"
  r53_zone_id = "${module.base.r53}"
  security_groups = "${aws_security_group.zk_sec.id}"
  source = "../lib/lib.20151013"
  subnet_id = "${module.common.data_a}"
  twitch_environment = "${var.twitch_environment}"
  vpc = "${module.common.vpc_id}"
  exhibitor_s3_config_file = "twitch-web-prod:config/exhibitor/${module.common.vpc_id}/exhibitor.properties"
  statsd_cluster_name = "${var.statsd_cluster_name}"
  statsd_server = "${var.statsd_server}"
  alert_interval = "${var.alert_interval}"
  send_alert = "${var.send_alert}"
}

module "zk_3" {
  twitch_aws_access_key = "${var.twitch_aws_access_key}"
  twitch_aws_secret_key = "${var.twitch_aws_secret_key}"
  twitch_web_aws_access_key = "${var.twitch_web_aws_access_key}"
  twitch_web_aws_secret_key = "${var.twitch_web_aws_secret_key}"

  env = "${var.env}"
  instance_type = "${var.ec2_instance_size}"
  r53_zone_id = "${module.base.r53}"
  security_groups = "${aws_security_group.zk_sec.id}"
  source = "../lib/lib.20151013"
  subnet_id = "${module.common.data_b}"
  twitch_environment = "${var.twitch_environment}"
  vpc = "${module.common.vpc_id}"
  exhibitor_s3_config_file = "twitch-web-prod:config/exhibitor/${module.common.vpc_id}/exhibitor.properties"
  statsd_cluster_name = "${var.statsd_cluster_name}"
  statsd_server = "${var.statsd_server}"
  alert_interval = "${var.alert_interval}"
  send_alert = "${var.send_alert}"
}

module "zk_4" {
  twitch_aws_access_key = "${var.twitch_aws_access_key}"
  twitch_aws_secret_key = "${var.twitch_aws_secret_key}"
  twitch_web_aws_access_key = "${var.twitch_web_aws_access_key}"
  twitch_web_aws_secret_key = "${var.twitch_web_aws_secret_key}"

  env = "${var.env}"
  instance_type = "${var.ec2_instance_size}"
  r53_zone_id = "${module.base.r53}"
  security_groups = "${aws_security_group.zk_sec.id}"
  source = "../lib/lib.20151013"
  subnet_id = "${module.common.data_c}"
  twitch_environment = "${var.twitch_environment}"
  vpc = "${module.common.vpc_id}"
  exhibitor_s3_config_file = "twitch-web-prod:config/exhibitor/${module.common.vpc_id}/exhibitor.properties"
  statsd_cluster_name = "${var.statsd_cluster_name}"
  statsd_server = "${var.statsd_server}"
  alert_interval = "${var.alert_interval}"
  send_alert = "${var.send_alert}"
}
