import sys
import os
import shutil
import argparse
import configparser
import logging
import json
from typing import Dict, Any
from zenyatta.aws import get_instance, get_aws_metadata, boto_client
from zenyatta.aws.s3 import upload_file_to_s3
from zenyatta.aws.emr import EMR
from zenyatta.common.util import parse_config_file, dump_to_json, dump_to_file, run_command, make_dir
from zenyatta.common.errors import ZenyattaError
from zenyatta.common import get_connections


def get_emr_launch_configuration(emr_name, aws_region, subnet_id, zenyatta_home, hardware_cfg_name,
                                 airflow_cfg, tags, team, env) -> Dict:
    try:
        config = configparser.SafeConfigParser()
        config.read(zenyatta_home + airflow_cfg)
        s3_bucket = get_connections()['aws'].get('s3-input').get('bucket')
        emr_cfg = parse_config_file(zenyatta_home + hardware_cfg_name)['emr']
        emr_settings = {}
        emr_settings['s3_bucket'] = s3_bucket
        emr_settings['bootstrap_path'] = emr_cfg['bootstrap'][0]['Path']
        emr_settings['SparkHome'] = config.get('spark', 'spark_home')
        emr_settings['Name'] = emr_name
        emr_settings['LogUri'] = 's3://{s3_bucket}/'.format(**locals()) + '{log_obj}'.format(**emr_cfg)
        emr_settings['ReleaseLabel'] = emr_cfg['release_label']
        emr_settings['InstanceAttribute'] = emr_cfg['instance_attribute']
        emr_settings['InstanceAttribute']['SubnetId'] = subnet_id
        emr_settings['InstanceAttribute']['InstanceProfile'] = 'rds-buddy-emr-ec2-' + team + '-' + env
        emr_settings['InstanceAttribute']['KeyName'] = team + '-rds-buddy-store-' + env + '-emr'
        emr_settings['InstanceGroups'] = emr_cfg['InstanceGroups']
        emr_settings['BootstrapActions'] = emr_cfg['bootstrap']
        emr_settings['BootstrapActions'][0]['Path'] = 's3://{s3_bucket}/'.format(**locals()) + \
            emr_settings['BootstrapActions'][0]['Path']
        app_str = ""
        for app in emr_cfg['applications']:
            app_str += "Name={Name} ".format(**app)
        emr_settings['Applications'] = app_str.strip()
        emr_settings['Configurations'] = emr_cfg['configurations']
        emr_settings['ServiceRole'] = 'rds-buddy-emr-' + team + '-' + env
        emr_settings['region'] = aws_region
        emr_settings['tags'] = (" ").join([(tag['Key'] + "=" + "\"" + tag['Value'] + "\"") for tag in tags])
        set_emr_name(zenyatta_home + airflow_cfg, emr_name)
        return emr_settings
    except Exception as e:
        raise ZenyattaError("buddy failed to read or make emr configuration. {}".format(e))


def launch_emr_cluster_aws_cli(emr_settings: Dict) -> str:
    emr_home = emr_settings['SparkHome'] + "/emr/"
    emr_instances_json = emr_home + 'buddy-emr-instances.json'
    emr_configuration_json = emr_home + 'buddy-emr-configuration.json'
    emr_ec2_attribute_json = emr_home + 'buddy-emr-ec2-attribute.json'
    emr_bootstrap_json = emr_home + 'buddy-emr-bootstrap.json'
    launch_emr_script = emr_home + 'buddy-launch-emr.sh'
    make_dir(emr_home)
    dump_to_json(emr_settings['InstanceGroups'], emr_instances_json)
    dump_to_json(emr_settings['Configurations'], emr_configuration_json)
    dump_to_json(emr_settings['InstanceAttribute'], emr_ec2_attribute_json)
    dump_to_json(emr_settings['BootstrapActions'], emr_bootstrap_json)
    try:
        aws_path = run_command(['which', 'aws']).decode().strip()
    except:
        raise ZenyattaError("aws cli is not found")
    cmd = [aws_path, "emr", "create-cluster",
           "--applications", "{Applications}".format(**emr_settings),
           "--tags", "{tags}".format(**emr_settings),
           "--ec2-attributes", "file://{emr_ec2_attribute_json}".format(**locals()),
           "--release-label", "{ReleaseLabel}".format(**emr_settings),
           "--log-uri", "{LogUri}".format(**emr_settings),
           "--instance-groups", "file://{emr_instances_json}".format(**locals()),
           "--configurations", "file://{emr_configuration_json}".format(**locals()),
           "--bootstrap-actions", "file://{emr_bootstrap_json}".format(**locals()),
           "--service-role", "{ServiceRole}".format(**emr_settings),
           "--enable-debugging",
           "--name", "{Name}".format(**emr_settings),
           "--region", "{region}".format(**emr_settings)]
    content = ' '.join(cmd)
    header = '#!/bin/bash'
    content = '\n'.join([header, content])
    dump_to_file(content, launch_emr_script)
    os.chmod(launch_emr_script, 0o777)
    rtn = run_command([launch_emr_script], env=None).decode()
    if "ClusterId" in rtn:
        cluster_id = json.loads(rtn)['ClusterId']
        shutil.rmtree(emr_home, ignore_errors=True)
        return cluster_id
    else:
        raise ZenyattaError("buddy failed to launch EMR cluster. msg: {}".format(rtn))


def set_emr_name(cfg_file_path: str, emr_name: str) -> str:
    if not os.path.isfile(cfg_file_path):
        raise Exception("could not locate airflow config file {}".format(cfg_file_path))
    try:
        config = configparser.SafeConfigParser()
        config.read(cfg_file_path)
        config.set('spark', 'emr_name', emr_name)
        with open(cfg_file_path, 'w') as f:
            config.write(f)
        logging.info('emr name is added.')
        return emr_name
    except Exception as e:
        raise ZenyattaError("buddy failed to add emr name in airflow.cfg. {}".format(e))


def sync_emr_config_to_s3(emr_settings: Dict) -> bool:
    s3_bucket = emr_settings['s3_bucket']
    s3_bootstrap_path = emr_settings['bootstrap_path']
    bootstrap_file = s3_bootstrap_path.split('/')[-1]
    s3_jar_path = 'zenyatta/spark/emr/jars'
    local_temp = '/opt/twitch/zenyatta/current/jars'
    try:
        header = '#!/bin/bash'
        content = ["mkdir -p /home/hadoop/custom_jars", "cd /home/hadoop/custom_jars"]
        for jar in os.listdir(local_temp):
            content.append("aws s3 cp s3://{s3_bucket}/{s3_jar_path}/{jar} .".format(**locals()))
            resp = upload_file_to_s3(s3_bucket, local_temp + "/" + jar, s3_jar_path + "/" + jar, None)
        content = '\n'.join([header, '\n'.join(content)])
        dump_to_file(content, local_temp + "/" + bootstrap_file)
        resp = upload_file_to_s3(s3_bucket, local_temp + "/" + bootstrap_file, s3_bootstrap_path, None)
        return True
    except Exception as e:
        raise ZenyattaError("buddy failed to sync emr config to s3. {}".format(e))


def launch_emr(emr_name, aws_region, team, env) -> bool:
    zenyatta_home = '/etc/zenyatta/'
    hardware_cfg_name = 'hardware_cfg.yaml'
    airflow_cfg = 'airflow.cfg'
    try:
        logging.info("buddy is looking for EMR cluster with name {emr_name}".format(**locals()))
        emr = EMR(emr_name)
        cluster = emr.fetch_emr_cluster()
        logging.info("buddy found a valid EMR cluster: {Id} {Name}.".format(**cluster))
        set_emr_name(zenyatta_home + airflow_cfg, emr_name)
        return True
    except ZenyattaError:
        logging.info("buddy did not find active EMR and will launch one for {emr_name}".format(**locals()))
        instance = get_instance(aws_region)
        subnet_id = instance.subnet_id
        tags = [tag for tag in instance.tags if tag['Key'] in ['Name', 'Owner', 'Project']]
        ec2_client, _ = boto_client('ec2')
        emr_settings = get_emr_launch_configuration(emr_name, aws_region, subnet_id, zenyatta_home,
                                                    hardware_cfg_name, airflow_cfg, tags, team, env)
        if sync_emr_config_to_s3(emr_settings):
            cluster_id = launch_emr_cluster_aws_cli(emr_settings)
            logging.info("buddy launched EMR cluster: {cluster_id}".format(**locals()))
            return True
    except Exception as e:
        logging.info("unexpected error: {0}".format(e))
        raise


if __name__ == '__main__':
    # buddy needs to provide team name and region
    # python setup_emr.py -n teamA -r region
    parser = argparse.ArgumentParser(description="set up emr cluster for airflow")
    parser.add_argument('-n', '--name', required=True, dest='team', action='store',
                        help='buddy team name used as emr name')
    parser.add_argument('-r', '--region', required=True, dest='region', action='store',
                        help='aws region to launch emr cluster')
    parser.add_argument('-e', '--env', required=True, dest='env', action='store',
                        help='environment, staging or prod')
    args = parser.parse_args()
    if launch_emr('buddy-' + args.team + '-' + args.env, args.region, args.team, args.env):
        sys.exit(0)
    else:
        sys.exit(1)
