provider "aws" {
  alias = "twitch-aws"
  profile = "twitch-aws"
  region = "us-west-2"
}

provider "aws" {
  alias = "twitch-web-dev"
  profile = "twitch-web-dev"
  region = "us-west-2"
}

provider "aws" {
  alias = "twitch-community-moderation-aws"
  profile = "twitch-community-moderation-aws"
  region = "us-west-2"
}

provider "aws" {
  alias = "twitch-chat-aws"
  profile = "twitch-chat-aws"
  region = "us-west-2"
}

module "master" {
  source                  = "git::git+ssh://git@git-aws.internal.justin.tv/terraform-modules/aws-instance-no-profile.git?ref=v1.1"
  account                 = "twitch-web-dev"
  region                  = "us-west-2"
  environment             = "development"
  vpc_security_group_ids = ["${module.zenyatta-sg.security_group_ids}"]
  subnet_ids              = ["${var.subnet_ids}"]
  service                 = "zenyatta"
  role                    = "zenyatta-master-dev"
  ami                     = "ami-a66dccc6"
  instance_type           = "m4.large"
  puppet                  = "puppet"
  count                   = "1"
  owner                   = "${var.owner}"

  instance_profile = "${aws_iam_instance_profile.zenyatta_profile.id}"
}

module "workers" {
  source                  = "git::git+ssh://git@git-aws.internal.justin.tv/michlyon/aws-instance-no-profile.git?ref=ML_add_mounts"
  account                 = "twitch-web-dev"
  region                  = "us-west-2"
  environment             = "development"
  vpc_security_group_ids = ["${module.zenyatta-sg.security_group_ids}"]
  subnet_ids              = ["${var.subnet_ids}"]
  service                 = "zenyatta"
  role                    = "zenyatta-worker-dev"
  ami                     = "ami-a66dccc6"
  instance_type           = "m4.xlarge"
  puppet                  = "puppet"
  count                   = "${var.data_count}"
  owner                   = "${var.owner}"

  instance_profile = "${aws_iam_instance_profile.zenyatta_profile.id}"

    mounts                  = <<EOF
mounts:
 - [ /dev/xvdb, /mnt, "ext4", "defaults,nofail,comment=cloudconfig", "0", "2" ]
fs_setup:
   - label: None
     filesystem: 'ext4'
     device: '/dev/xvdb'
     partition: 'auto'
EOF
}

# RDS
module "zenyatta-postgres" {
  source                  = "../../modules/rds"
  account                 = "twitch-web-dev"
  environment             = "development"
  identifier              = "zenyatta-postgres"
  subnet_group_name       = "d8a-subnet-group"
  vpc_security_group_ids = ["${module.zenyatta-sg.security_group_ids}"]
  owner                   = "${var.owner}"
}


# SECURITY GROUP
module "zenyatta-sg" {
  source                  = "../../modules/security_group"
  service                 = "zenyatta"
  account                 = "twitch-web-dev"
  subnets                 = "${var.private_subnets}"
  vpc_id                  = "${var.vpc_id}"
  environment             = "development"
  owner                   = "${var.owner}"
}

# REDIS
module "zenyatta-redis" {
  source                = "../../modules/elasticache"
  account               = "twitch-web-dev"
  service               = "zenyatta"
  environment           = "development"
  parameter_group_name  = "default.redis2.8"
  subnet_ids            = "${var.subnet_ids}"
  security_group_ids = ["${module.zenyatta-sg.security_group_ids}"]
  owner                 = "${var.owner}"
}
# manual part

# Create some EBS volumes
variable "azs" {
  type = "list"
  default = ["us-west-2a","us-west-2c"]
}

# Attach the EBS volumes (AZ A)
resource "aws_ebs_volume" "data" {
  availability_zone = "us-west-2a"
  size = "${var.ebs_size}"
  type = "gp2"
  count = "${var.data_count}"
}

resource "aws_volume_attachment" "data" {
  device_name = "/dev/xvdb"
  volume_id = "${element(aws_ebs_volume.data.*.id, count.index)}"
  instance_id = "${element(module.workers.instance_ids, count.index)}"
  count = "${var.data_count}"
}

resource "aws_route53_record" "zenyatta" {
  provider = "aws.twitch-aws"
  count    = "${var.data_count}"
  zone_id  = "${var.zone_id}"
  name = "zenyatta.development.us-west2.justin.tv"
  type     = "CNAME"
  ttl      = "60"
  records  = ["${element(module.master.fqdns,count.index)}"]
}


resource "aws_iam_instance_profile" "zenyatta_profile" {
  provider = "aws.twitch-web-dev"
  name = "zenyatta"
  role = "${aws_iam_role.airflow.id}"
}

# moderation trust
resource "aws_iam_role" "airflow-moderation-development" {
  name = "airflow-dev"
  provider = "aws.twitch-community-moderation-aws"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
         "AWS": "arn:aws:iam::465369119046:root"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "airflow-moderation-development" {
  name = "airflow-dev"
  provider = "aws.twitch-community-moderation-aws"
  role = "${aws_iam_role.airflow-moderation-development.id}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
                "rds:*"
            ],
      "Resource": "*"
    }
  ]
}
EOF
}

# chat trust
resource "aws_iam_role" "airflow-chat-development" {
  name = "airflow-dev"
  provider = "aws.twitch-chat-aws"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
         "AWS": "arn:aws:iam::465369119046:root"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "airflow-chat-development" {
  name = "airflow-dev"
  provider = "aws.twitch-chat-aws"
  role = "${aws_iam_role.airflow-chat-development.id}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
                "s3:Get*",
                "s3:List*",
                "iam:GetRolePolicy"
            ],
      "Resource": [
              "arn:aws:s3:::d8a-chat-backup*",
              "arn:aws:s3:::d8a-chat-backup/*"
      ]
    }
  ]
}
EOF
}

# MASTER ROLE
resource "aws_iam_role" "airflow" {
  name = "airflow"
  provider = "aws.twitch-web-dev"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "ec2.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}


resource "aws_iam_role_policy" "d8a_airflow_policy" {
    name = "zenyatta-access"
    role = "${aws_iam_role.airflow.id}"
    provider = "aws.twitch-web-dev"
    policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "cloudwatch:*",
                "autoscaling:Describe",
                "logs:*",
                "ecr:*",
                "rds:*",
                "ec2:*",
                "iam:*",
                "sns:*",
                "elasticache:*",
                "elasticmapreduce:*"
            ],
            "Resource": "*"
        },
        {
            "Effect": "Allow",
            "Action": [
                "s3:Get*",
                "s3:Put*",
                "s3:List*"
            ],
            "Resource": [
                "arn:aws:s3:::twitch-web-dev*",
                "arn:aws:s3:::twitch-web-dev/*",
                "arn:aws:s3:::twitch-d8a-test*",
                "arn:aws:s3:::twitch-d8a-test/*",
                "arn:aws:s3:::twitch-d8a-etl/*",
                "arn:aws:s3:::science-input/dbsnapshots*",
                "arn:aws:s3:::science-input/dbsnapshots/*"
            ]
        },
        {
            "Effect": "Allow",
            "Action": [
                "s3:Get*",
                "s3:List*"
            ],
            "Resource": [
                "arn:aws:s3:::d8a-chat-backup*",
                "arn:aws:s3:::d8a-chat-backup/*",
                "arn:aws:s3:::d8a-web-backup*",
                "arn:aws:s3:::d8a-web-backup/*"
            ]
        },
        {
            "Effect": "Deny",
            "Action": [
                "cloudwatch:Delete*",
                "logs:Delete*",
                "sns:Delete*"
            ],
            "Resource": "*"
        },
        {
            "Effect": "Allow",
            "Action": "sts:AssumeRole",
            "Resource": [
              "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/airflow-development",
              "arn:aws:iam::465369119046:role/airflow",
              "arn:aws:iam::654663525958:role/airflow-dev",
              "arn:aws:iam::603200399373:role/airflow-dev"
            ]
        }
    ]
}
EOF
}
