provider "aws" {
  alias = "twitch-aws"
  profile = "twitch-aws"
  region = "us-west-2"
}

provider "aws" {
  alias = "twitch-web-aws"
  profile = "twitch-web-aws"
  region = "us-west-2"
}


provider "aws" {
  alias = "twitch-chat-aws"
  profile = "twitch-chat-aws"
  region = "us-west-2"
}


provider "aws" {
  alias = "twitch-science-aws"
  profile = "twitch-science-aws"
  region = "us-west-2"
}

module "master" {
  source                  = "git::git+ssh://git@git-aws.internal.justin.tv/terraform-modules/aws-instance-no-profile.git?ref=v1.1"
  account                 = "twitch-web-aws"
  region                  = "us-west-2"
  environment             = "production"
  vpc_security_group_ids = ["${module.zenyatta-sg.security_group_ids}"]
  subnet_ids              = ["${var.subnet_ids}"]
  service                 = "zenyatta"
  role                    = "zenyatta-master"
  ami                     = "${var.ami}"
  instance_type           = "m4.large"
  puppet                  = "puppet"
  count                   = "1"
  owner                   = "${var.owner}"

  instance_profile = "${aws_iam_instance_profile.zenyatta_profile.id}"
}

module "workers" {
  source                  = "git::git+ssh://git@git-aws.internal.justin.tv/michlyon/aws-instance-no-profile.git?ref=output_az"
  account                 = "twitch-web-aws"
  region                  = "us-west-2"
  environment             = "production"
  vpc_security_group_ids = ["${module.zenyatta-sg.security_group_ids}"]
  subnet_ids              = ["${var.subnet_ids}"]
  service                 = "zenyatta"
  role                    = "zenyatta-worker"
  ami                     = "${var.ami}"
  instance_type           = "m4.4xlarge"
  puppet                  = "puppet"
  count                   = "${var.data_count}"
  owner                   = "${var.owner}"
  mounts                  = <<EOF
mounts:
 - [ /dev/xvdb, /mnt, "ext4", "defaults,nofail,comment=cloudconfig", "0", "2" ]
fs_setup:
   - label: None
     filesystem: 'ext4'
     device: '/dev/xvdb'
     partition: 'auto'
EOF

  instance_profile = "${aws_iam_instance_profile.zenyatta_profile.id}"
}

# SECURITY GROUP
module "zenyatta-sg" {
  source                  = "../../modules/security_group"
  service                 = "zenyatta"
  account                 = "twitch-web-aws"
  subnets                 = "${var.private_subnets}"
  vpc_id                  = "${var.vpc_id}"
  environment             = "production"
  owner                   = "${var.owner}"
}

# manual part

# Create some EBS volumes
variable "azs" {
  type = "list"
  default = ["us-west-2a","us-west-2c"]
}

# Attach the EBS volumes (AZ A)
resource "aws_ebs_volume" "data" {
  //availability_zone = "us-west-2"
  availability_zone = "${element(module.workers.availability_zones, count.index)}"
  size = "${var.ebs_size}"
  type = "gp2"
  count = "${var.data_count}"

  lifecycle {
    ignore_changes = [ "volume", "instance" ]
  }
}

resource "aws_volume_attachment" "data" {
  device_name = "/dev/xvdb"
  volume_id = "${element(aws_ebs_volume.data.*.id, count.index)}"
  instance_id = "${element(module.workers.instance_ids, count.index)}"
  count = "${var.data_count}"

  lifecycle {
    ignore_changes = [ "volume", "instance" ]
  }
}

resource "aws_route53_record" "zenyatta" {
  provider = "aws.twitch-aws"
  zone_id  = "${var.zone_id}"
  name     = "zenyatta.production.twitch-web-aws.us-west2.justin.tv"
  type     = "CNAME"
  ttl      = "60"
  records  = ["${module.master.fqdns}"]
}

resource "aws_iam_instance_profile" "zenyatta_profile" {
  provider = "aws.twitch-web-aws"
  name = "zenyatta"
  role = "${aws_iam_role.airflow.id}"
}

# chat trust
resource "aws_iam_role" "airflow-chat" {
  name = "airflow-prod"
  provider = "aws.twitch-chat-aws"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
         "AWS": "arn:aws:iam::641044725657:root"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "airflow-chat" {
  name = "airflow-prod"
  provider = "aws.twitch-chat-aws"
  role = "${aws_iam_role.airflow-chat.id}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
                "s3:Get*",
                "s3:List*",
                "iam:GetRolePolicy"
            ],
      "Resource": [
              "arn:aws:s3:::d8a-chat-backup*",
              "arn:aws:s3:::d8a-chat-backup/*"
      ]
    }
  ]
}
EOF
}

# chat trust
resource "aws_iam_role" "airflow-science" {
  name = "zenyatta-prod"
  provider = "aws.twitch-science-aws"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
         "AWS": [
            "arn:aws:iam::641044725657:root"
          ]
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "airflow-science" {
  name = "zenyatta-prod"
  provider = "aws.twitch-science-aws"
  role = "${aws_iam_role.airflow-science.id}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
                "s3:Get*",
                "s3:List*",
                "s3:Put*",
                "iam:GetRolePolicy"
            ],
      "Resource": [
          "arn:aws:s3:::science-input*",
          "arn:aws:s3:::science-input/*"
      ]
    }
  ]
}
EOF
}

# MASTER ROLE
resource "aws_iam_role" "airflow" {
  name = "airflow"
  provider = "aws.twitch-web-aws"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "ec2.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "d8a_airflow_policy" {
    name = "zenyatta-access"
    role = "${aws_iam_role.airflow.id}"
    provider = "aws.twitch-web-aws"
    policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
               "cloudwatch:*",
                "autoscaling:Describe",
                "logs:*",
                "ecr:*",
                "sns:*",
                "rds:*",
                "ec2:*",
                "iam:*",
                "sts:DecodeAuthorizationMessage",
                "s3:*",
                "elasticache:*",
                "elasticmapreduce:*"
            ],
            "Resource": "*"
        },
        {
            "Effect": "Allow",
            "Action": [
                "s3:Get*",
                "s3:Put*",
                "s3:List*"
            ],
            "Resource": [
                "arn:aws:s3:::science-input*",
                "arn:aws:s3:::science-input/*"

            ]
        },
        {
            "Effect": "Allow",
            "Action": [
                "s3:Get*",
                "s3:List*"
            ],
            "Resource": [
                "arn:aws:s3:::d8a-web-backup*",
                "arn:aws:s3:::d8a-web-backup/*",
                "arn:aws:s3:::d8a-chat-backup*",
                "arn:aws:s3:::d8a-chat-backup/*"
            ]
        },
        {
            "Effect": "Deny",
            "Action": [
                "cloudwatch:Delete*",
                "logs:Delete*",
                "sns:Delete*"
            ],
            "Resource": "*"
        },
        {
            "Effect": "Allow",
            "Action": "sts:AssumeRole",
            "Resource": [
              "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/airflow-production",
              "arn:aws:iam::641044725657:role/airflow",
              "arn:aws:iam::654663525958:role/airflow-prod",
              "arn:aws:iam::603200399373:role/airflow-prod",
              "arn:aws:iam::673385534282:role/zenyatta-prod",
              "arn:aws:iam::134912696936:role/airflow",
              "arn:aws:iam::641044725657:role/zenyatta-emr",
              "arn:aws:iam::989470033077:role/airflow"
            ]
        }
    ]
}
EOF
}
