from datetime import datetime
from dateutil.tz import tzutc

import sys
import os

# remember python3 supports mocks
from unittest.mock import MagicMock
from tests.static.constants import DBInstanceIdentifier

PARENT = '../'
SCRIPT_DIR = os.path.dirname(os.path.realpath(os.path.join(os.getcwd(), os.path.expanduser(__file__))))
sys.path.append(os.path.normpath(os.path.join(SCRIPT_DIR, PARENT)))

# begin local imports
# test imports

DESCRIBE_DB_INSTANCES = {
    'DBInstances': [{
        'AllocatedStorage': 150,
        'AutoMinorVersionUpgrade': True,
        'AvailabilityZone': 'us-west-2a',
        'BackupRetentionPeriod': 2,
        'CACertificateIdentifier': 'rds-ca-2015',
        'CopyTagsToSnapshot': False,
        'DBInstanceClass': 'db.m3.xlarge',
        'DBInstanceIdentifier': DBInstanceIdentifier,
        'DBInstanceStatus': 'available',
        'DBName': 'discovery',
        'DBParameterGroups': [{'DBParameterGroupName': 'discovery-staging-95',
                               'ParameterApplyStatus': 'in-sync'}],
        'DBSecurityGroups': [],
        'DBSubnetGroup': {'DBSubnetGroupDescription': 'D8A Subnets',
                          'DBSubnetGroupName': 'discovery-staging-subnet',
                          'SubnetGroupStatus': 'Complete',
                          'Subnets': [{'SubnetAvailabilityZone': {'Name': 'us-west-2a'},
                                       'SubnetIdentifier': 'subnet-b1af88d4',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2b'},
                                       'SubnetIdentifier': 'subnet-1d76336a',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2c'},
                                       'SubnetIdentifier': 'subnet-cbd54a92',
                                       'SubnetStatus': 'Active'}],
                          'VpcId': 'vpc-230f5e46'},
        'DbInstancePort': 0,
        'DbiResourceId': 'db-2N3Z3IKLN2KCIPXSIRTSGVYGJE',
        'DomainMemberships': [],
        'Endpoint': {'Address': DBInstanceIdentifier+".cifgffw7w2ar.us-west-2.rds.amazonaws.com",
                     'HostedZoneId': 'Z1PVIF0B656C1W',
                     'Port': 5432},
        'Engine': 'postgres',
        'EngineVersion': '9.5.2',
        'InstanceCreateTime': datetime(2016, 12, 14, 18, 18, 55, 498000,
                                       tzinfo=tzutc()),
        'LatestRestorableTime': datetime(2016, 12, 21, 19, 24, 27,
                                         tzinfo=tzutc()),
        'LicenseModel': 'postgresql-license',
        'MasterUsername': 'super',
        'MonitoringInterval': 0,
        'MultiAZ': False,
        'OptionGroupMemberships': [{'OptionGroupName': 'default:postgres-9-5',
                                    'Status': 'in-sync'}],
        'PendingModifiedValues': {},
        'PreferredBackupWindow': '10:24-10:54',
        'PreferredMaintenanceWindow': 'wed:07:40-wed:08:10',
        'PubliclyAccessible': False,
        'ReadReplicaDBInstanceIdentifiers': [],
        'StorageEncrypted': False,
        'StorageType': 'gp2',
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-17005271'}]}],
        'ResponseMetadata': {'HTTPStatusCode': 200,
                             'RequestId': '812aae51-c7b3-11e6-a72c-31e313ed89bf'}}


DELETE_DB_INSTANCE = {
    'DBInstance': {
        'AllocatedStorage': 150,
        'AutoMinorVersionUpgrade': True,
        'AvailabilityZone': 'us-west-2a',
        'BackupRetentionPeriod': 2,
        'CACertificateIdentifier': 'rds-ca-2015',
        'CopyTagsToSnapshot': False,
        'DBInstanceClass': 'db.m3.xlarge',
        'DBInstanceIdentifier': DBInstanceIdentifier,
        'DBInstanceStatus': 'deleting',
        'DBName': 'discovery',
        'DBParameterGroups': [{'DBParameterGroupName': 'discovery-staging-95',
                               'ParameterApplyStatus': 'in-sync'}],
        'DBSecurityGroups': [],
        'DBSubnetGroup': {'DBSubnetGroupDescription': 'D8A Subnets',
                          'DBSubnetGroupName': 'discovery-staging-subnet',
                          'SubnetGroupStatus': 'Complete',
                          'Subnets': [{'SubnetAvailabilityZone': {'Name': 'us-west-2a'},
                                       'SubnetIdentifier': 'subnet-b1af88d4',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2b'},
                                       'SubnetIdentifier': 'subnet-1d76336a',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2c'},
                                       'SubnetIdentifier': 'subnet-cbd54a92',
                                       'SubnetStatus': 'Active'}],
                          'VpcId': 'vpc-230f5e46'},
        'DbInstancePort': 0,
        'DbiResourceId': 'db-WYRJGDE2TANN5AVKKGQSIQC7ZM',
        'DomainMemberships': [],
        'Endpoint': {'Address': DBInstanceIdentifier+'.cifgffw7w2ar.us-west-2.rds.amazonaws.com',
                     'HostedZoneId': 'Z1PVIF0B656C1W',
                     'Port': 5432},
        'Engine': 'postgres',
        'EngineVersion': '9.5.2',
        'InstanceCreateTime': datetime(2016, 12, 21, 19, 41, 38, 156000, tzinfo=tzutc()),
        'LatestRestorableTime': datetime(2016, 12, 21, 20, 25, 59, tzinfo=tzutc()),
        'LicenseModel': 'postgresql-license',
        'MasterUsername': 'super',
        'MonitoringInterval': 0,
        'MultiAZ': False,
        'OptionGroupMemberships': [{'OptionGroupName': 'default:postgres-9-5',
                                    'Status': 'in-sync'}],
        'PendingModifiedValues': {},
        'PreferredBackupWindow': '10:24-10:54',
        'PreferredMaintenanceWindow': 'wed:07:40-wed:08:10',
        'PubliclyAccessible': False,
        'ReadReplicaDBInstanceIdentifiers': [],
        'StorageEncrypted': False,
        'StorageType': 'gp2',
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-c56b03a1'}]},
        'ResponseMetadata': {'HTTPStatusCode': 200,
                             'RequestId': '0787fbdb-c7bc-11e6-acb2-8726f5590fad'}}


DB_RESTORE_TO_POINT_IN_TIME = {
    'DBInstance': {
        'AllocatedStorage': 150,
        'AutoMinorVersionUpgrade': True,
        'BackupRetentionPeriod': 2,
        'CACertificateIdentifier': 'rds-ca-2015',
        'CopyTagsToSnapshot': False,
        'DBInstanceClass': 'db.m3.xlarge',
        'DBInstanceIdentifier': DBInstanceIdentifier,
        'DBInstanceStatus': 'creating',
        'DBName': 'discovery',
        'DBParameterGroups': [{'DBParameterGroupName': 'discovery-staging-95',
                               'ParameterApplyStatus': 'in-sync'}],
        'DBSecurityGroups': [],
        'DBSubnetGroup': {'DBSubnetGroupDescription': 'D8A Subnets',
                          'DBSubnetGroupName': 'discovery-staging-subnet',
                          'SubnetGroupStatus': 'Complete',
                          'Subnets': [{'SubnetAvailabilityZone': {'Name': 'us-west-2a'},
                                       'SubnetIdentifier': 'subnet-b1af88d4',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2b'},
                                       'SubnetIdentifier': 'subnet-1d76336a',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2c'},
                                       'SubnetIdentifier': 'subnet-cbd54a92',
                                       'SubnetStatus': 'Active'}],
                          'VpcId': 'vpc-230f5e46'},
        'DbInstancePort': 0,
        'DbiResourceId': 'db-CEW6APC6MJAJX2S2KP3QRIIS5U',
        'DomainMemberships': [],
        'Engine': 'postgres',
        'EngineVersion': '9.5.2',
        'LicenseModel': 'postgresql-license',
        'MasterUsername': 'super',
        'MonitoringInterval': 0,
        'MultiAZ': False,
        'OptionGroupMemberships': [{'OptionGroupName': 'default:postgres-9-5',
                                    'Status': 'pending-apply'}],
        'PendingModifiedValues': {},
        'PreferredBackupWindow': '10:24-10:54',
        'PreferredMaintenanceWindow': 'wed:07:40-wed:08:10',
        'PubliclyAccessible': False,
        'ReadReplicaDBInstanceIdentifiers': [],
        'StorageEncrypted': False,
        'StorageType': 'gp2',
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-c56b03a1'}]},
        'ResponseMetadata': {'HTTPStatusCode': 200,
                             'RequestId': '75f26e31-c7bc-11e6-98fe-91120c7f3a4c'}}


DB_STATUS_CREATING = {
    'DBInstances': [{
        'AllocatedStorage': 150,
        'AutoMinorVersionUpgrade': True,
        'AvailabilityZone': 'us-west-2c',
        'BackupRetentionPeriod': 2,
        'CACertificateIdentifier': 'rds-ca-2015',
        'CopyTagsToSnapshot': False,
        'DBInstanceClass': 'db.m3.xlarge',
        'DBInstanceIdentifier': DBInstanceIdentifier,
        'DBInstanceStatus': 'creating',
        'DBName': 'discovery',
        'DBParameterGroups': [{'DBParameterGroupName': 'discovery-staging-95',
                               'ParameterApplyStatus': 'in-sync'}],
        'DBSecurityGroups': [],
        'DBSubnetGroup': {'DBSubnetGroupDescription': 'D8A Subnets',
                          'DBSubnetGroupName': 'discovery-staging-subnet',
                          'SubnetGroupStatus': 'Complete',
                          'Subnets': [{'SubnetAvailabilityZone': {'Name': 'us-west-2a'},
                                       'SubnetIdentifier': 'subnet-b1af88d4',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2b'},
                                       'SubnetIdentifier': 'subnet-1d76336a',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2c'},
                                       'SubnetIdentifier': 'subnet-cbd54a92',
                                       'SubnetStatus': 'Active'}],
                          'VpcId': 'vpc-230f5e46'},
        'DbInstancePort': 0,
        'DbiResourceId': 'db-CEW6APC6MJAJX2S2KP3QRIIS5U',
        'DomainMemberships': [],
        'Engine': 'postgres',
        'EngineVersion': '9.5.2',
        'LicenseModel': 'postgresql-license',
        'MasterUsername': 'super',
        'MonitoringInterval': 0,
        'MultiAZ': False,
        'OptionGroupMemberships': [{'OptionGroupName': 'default:postgres-9-5',
                                    'Status': 'pending-apply'}],
        'PendingModifiedValues': {},
        'PreferredBackupWindow': '10:24-10:54',
        'PreferredMaintenanceWindow': 'wed:07:40-wed:08:10',
        'PubliclyAccessible': False,
        'ReadReplicaDBInstanceIdentifiers': [],
        'StorageEncrypted': False,
        'StorageType': 'gp2',
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-c56b03a1'}]}],
        'ResponseMetadata': {'HTTPStatusCode': 200,
                             'RequestId': 'c109c239-c7bc-11e6-8c87-bd16816aee1b'}}


MODIFY_DB_INSTANCE = {
    'DBInstance': {
        'AllocatedStorage': 150,
        'AutoMinorVersionUpgrade': True,
        'AvailabilityZone': 'us-west-2c',
        'BackupRetentionPeriod': 2,
        'CACertificateIdentifier': 'rds-ca-2015',
        'CopyTagsToSnapshot': False,
        'DBInstanceClass': 'db.m3.xlarge',
        'DBInstanceIdentifier': DBInstanceIdentifier,
        'DBInstanceStatus': 'available',
        'DBName': 'discovery',
        'DBParameterGroups': [{'DBParameterGroupName': 'discovery-staging-95',
                               'ParameterApplyStatus': 'in-sync'}],
        'DBSecurityGroups': [],
        'DBSubnetGroup': {'DBSubnetGroupDescription': 'D8A Subnets',
                          'DBSubnetGroupName': 'discovery-staging-subnet',
                          'SubnetGroupStatus': 'Complete',
                          'Subnets': [{'SubnetAvailabilityZone': {'Name': 'us-west-2a'},
                                       'SubnetIdentifier': 'subnet-b1af88d4',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2b'},
                                       'SubnetIdentifier': 'subnet-1d76336a',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2c'},
                                       'SubnetIdentifier': 'subnet-cbd54a92',
                                       'SubnetStatus': 'Active'}],
                          'VpcId': 'vpc-230f5e46'},
        'DbInstancePort': 0,
        'DbiResourceId': 'db-CEW6APC6MJAJX2S2KP3QRIIS5U',
        'DomainMemberships': [],
        'Endpoint': {'Address': DBInstanceIdentifier+'.cifgffw7w2ar.us-west-2.rds.amazonaws.com',
                     'HostedZoneId': 'Z1PVIF0B656C1W',
                     'Port': 5432},
        'Engine': 'postgres',
        'EngineVersion': '9.5.2',
        'InstanceCreateTime': datetime(2016, 12, 21, 20, 34, 56, 600000, tzinfo=tzutc()),
        'LatestRestorableTime': datetime(2016, 12, 21, 21, 7, 44, tzinfo=tzutc()),
        'LicenseModel': 'postgresql-license',
        'MasterUsername': 'super',
        'MonitoringInterval': 0,
        'MultiAZ': False,
        'OptionGroupMemberships': [{'OptionGroupName': 'default:postgres-9-5',
                                    'Status': 'in-sync'}],
        'PendingModifiedValues': {'BackupRetentionPeriod': 0},
        'PreferredBackupWindow': '10:24-10:54',
        'PreferredMaintenanceWindow': 'wed:07:40-wed:08:10',
        'PubliclyAccessible': False,
        'ReadReplicaDBInstanceIdentifiers': [],
        'StorageEncrypted': False,
        'StorageType': 'gp2',
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-c56b03a1'}]},
        'ResponseMetadata': {'HTTPStatusCode': 200,
                             'RequestId': 'f566f6bd-c7c1-11e6-98fe-91120c7f3a4c'}}


def mock_destroy_resource():
    def delete(**kwargs):
        cp = DELETE_DB_INSTANCE.copy()
        cp['DBInstance']['DBInstanceIdentifier'] = kwargs.get('DBInstanceIdentifier')
        return cp

    def describe(**kwargs):
        cp = DESCRIBE_DB_INSTANCES.copy()
        cp['DBInstances'] = []
        return cp

    mock = MagicMock()
    mock.delete_db_instance = MagicMock(side_effect=delete)
    mock.describe_db_instances = MagicMock(side_effect=describe)
    return mock, None


def mocked_boto_rds_instance():

    def delete_sides(**kwargs):
        """
        boto.delete_db_instance(DBInstanceIdentifier=self.identifier,
        SkipFinalSnapshot=True)
        """
        cp = DELETE_DB_INSTANCE.copy()
        cp['DBInstance']['DBInstanceIdentifier'] = kwargs.get('DBInstanceIdentifier')
        return cp

    def restore_sides(**kwargs):
        """
        boto.restore_db_instance_to_point_in_time(SourceDBInstanceIdentifier=self.identifier,
                                                         TargetDBInstanceIdentifier=target.identifier,
                                                         RestoreTime=point_in_time,
                                                         DBSubnetGroupName=self.subnet(),
                                                         Tags=[{'Key': 'project', 'Value': 'zenyatta'}])
        """
        cp = DB_RESTORE_TO_POINT_IN_TIME.copy()
        cp['DBInstance']['DBInstanceIdentifier'] = kwargs.get('TargetDBInstanceIdentifier')
        cp['DBInstance']['DBSubnetGroup']['DBSubnetGroupName'] = kwargs.get('DBSubnetGroupName')
        return cp

    def modify_sides(**kwargs):
        """
        boto.modify_db_instance(
            DBInstanceIdentifier=target.identifier,
            BackupRetentionPeriod=0,  # means no backups for these dbs
            VpcSecurityGroupIds=[self.security_group()])
        """
        cp = MODIFY_DB_INSTANCE.copy()
        cp['DBInstance']['DBInstanceIdentifier'] = kwargs.get('DBInstanceIdentifier')
        cp['DBInstance']['PendingModifiedValues']['BackupRetentionPeriod'] = \
            kwargs.get('BackupRetentionPeriod')
        cp['DBInstance']['PendingModifiedValues']['DBSecurityGroups'] = kwargs.get('VpcSecurityGroupIds')
        return cp

    def describe_sides(**kwargs):
        """
        boto.describe_db_instances(DBInstanceIdentifier=self.identifier)['DBInstances'][0]
        """
        cp = DESCRIBE_DB_INSTANCES.copy()
        cp['DBInstances'][0]['DBInstanceIdentifier'] = kwargs.get('DBInstanceIdentifier')
        return cp

    mock = MagicMock()
    mock.delete_db_instance = MagicMock(side_effect=delete_sides)
    mock.restore_db_instance_to_point_in_time = MagicMock(side_effect=restore_sides)
    mock.modify_db_instance = MagicMock(side_effect=modify_sides)

    mock.describe_db_instances = MagicMock(side_effect=describe_sides)
    return mock, None

# rds.describe_db_clusters(DBClusterIdentifier='cohesion-chat')
DESCRIBE_DB_CLUSTERS = {
    'DBClusters': [{
        'AllocatedStorage': 1,
        'AvailabilityZones': ['us-west-2a', 'us-west-2b', 'us-west-2c'],
        'BackupRetentionPeriod': 5,
        'DBClusterIdentifier': 'cohesion-chat',
        'DBClusterMembers': [{'DBClusterParameterGroupStatus': 'in-sync',
                              'DBInstanceIdentifier': 'cohesion-chat-0',
                              'IsClusterWriter': True,
                              'PromotionTier': 0},
                             {'DBClusterParameterGroupStatus': 'in-sync',
                              'DBInstanceIdentifier': 'cohesion-chat-2',
                              'IsClusterWriter': False,
                              'PromotionTier': 0},
                             {'DBClusterParameterGroupStatus': 'in-sync',
                              'DBInstanceIdentifier': 'cohesion-chat-1',
                              'IsClusterWriter': False,
                              'PromotionTier': 0}],
        'DBClusterParameterGroup': 'default.aurora5.6',
        'DBSubnetGroup': 'd8a-cohesion-subnet-group',
        'DatabaseName': 'cohesion',
        'DbClusterResourceId': 'cluster-5YBH355343NSWI7S4VKQWN35XY',
        'EarliestRestorableTime': datetime(2016, 12, 22, 7, 13, 15, 505000, tzinfo=tzutc()),
        'Endpoint': 'cohesion-chat.cluster-cifgffw7w2ar.us-west-2.rds.amazonaws.com',
        'Engine': 'aurora',
        'EngineVersion': '5.6.10a',
        'HostedZoneId': 'Z1PVIF0B656C1W',
        'LatestRestorableTime': datetime(2016, 12, 27, 19, 36, 8, 961000, tzinfo=tzutc()),
        'MasterUsername': 'cohesion',
        'Port': 3306,
        'PreferredBackupWindow': '07:00-09:00',
        'PreferredMaintenanceWindow': 'sat:10:48-sat:11:18',
        'Status': 'available',
        'StorageEncrypted': False,
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-c06b03a4'}]}],
        'ResponseMetadata': {'HTTPStatusCode': 200,
                             'RequestId': '1b8b7f91-cc6c-11e6-a26d-c1b334c41e17'}}

# rds.restore_db_cluster_to_point_in_time(
#   SourceDBClusterIdentifier='cohesion-chat', DBClusterIdentifier='cohesion-chat-test',
#   RestoreToTime=datetime.now(), VpcSecurityGroupIds=['sg-c06b03a4'],
#   DBSubnetGroupName='d8a-cohesion-subnet-group', Tags=[{'Key': 'project', 'Value': 'zenyatta'}])
DB_CLUSTER_RESTORE_TO_POINT_IN_TIME = {
    'DBCluster': {
        'AllocatedStorage': 1,
        'AvailabilityZones': ['us-west-2a', 'us-west-2b', 'us-west-2c'],
        'BackupRetentionPeriod': 5,
        'DBClusterIdentifier': 'cohesion-chat-test',
        'DBClusterMembers': [],
        'DBClusterParameterGroup': 'default.aurora5.6',
        'DBSubnetGroup': 'd8a-cohesion-subnet-group',
        'DatabaseName': 'cohesion',
        'DbClusterResourceId': 'cluster-U2V2QLSEOKYEYEYJAEY5KUBNQI',
        'Endpoint': 'cohesion-chat-test.cluster-cifgffw7w2ar.us-west-2.rds.amazonaws.com',
        'Engine': 'aurora',
        'EngineVersion': '5.6.10a',
        'HostedZoneId': 'Z1PVIF0B656C1W',
        'MasterUsername': 'cohesion',
        'Port': 3306,
        'PreferredBackupWindow': '07:00-09:00',
        'PreferredMaintenanceWindow': 'sat:10:48-sat:11:18',
        'Status': 'creating',
        'StorageEncrypted': False,
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-c06b03a4'}]},
        'ResponseMetadata': {'HTTPStatusCode': 200,
                             'RequestId': 'a6d3a0c4-cc6c-11e6-b861-371bf5988138'}}


MODIFY_DB_CLUSTER = {
    'DBCluster': {
        'AllocatedStorage': 1,
        'AvailabilityZones': ['us-west-2a', 'us-west-2b', 'us-west-2c'],
        'BackupRetentionPeriod': 1,
        'DBClusterIdentifier': 'cohesion-chat-test',
        'DBClusterMembers': [],
        'DBClusterParameterGroup': 'default.aurora5.6',
        'DBSubnetGroup': 'd8a-cohesion-subnet-group',
        'DatabaseName': 'cohesion',
        'DbClusterResourceId': 'cluster-U2V2QLSEOKYEYEYJAEY5KUBNQI',
        'EarliestRestorableTime': datetime(2016, 12, 27, 20, 26, 43, 647000, tzinfo=tzutc()),
        'Endpoint': 'cohesion-chat-test.cluster-cifgffw7w2ar.us-west-2.rds.amazonaws.com',
        'Engine': 'aurora',
        'EngineVersion': '5.6.10a',
        'HostedZoneId': 'Z1PVIF0B656C1W',
        'LatestRestorableTime': datetime(2016, 12, 27, 20, 26, 43, 647000, tzinfo=tzutc()),
        'MasterUsername': 'cohesion',
        'Port': 3306,
        'PreferredBackupWindow': '07:00-09:00',
        'PreferredMaintenanceWindow': 'sat:10:48-sat:11:18',
        'Status': 'available',
        'StorageEncrypted': False,
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-c06b03a4'}]},
        'ResponseMetadata': {'HTTPStatusCode': 200,
                             'RequestId': '46465e61-cc77-11e6-b13a-bf94e7c256a7'}}

CREATE_DB_INSTANCE = {
    'DBInstance': {
        'AllocatedStorage': 1,
        'AutoMinorVersionUpgrade': True,
        'BackupRetentionPeriod': 1,
        'CACertificateIdentifier': 'rds-ca-2015',
        'CopyTagsToSnapshot': False,
        'DBClusterIdentifier': 'cohesion-chat-test',
        'DBInstanceClass': 'db.r3.8xlarge',
        'DBInstanceIdentifier': 'cohesion-chat-test-reader',
        'DBInstanceStatus': 'creating',
        'DBName': 'cohesion',
        'DBParameterGroups': [{'DBParameterGroupName': 'default.aurora5.6',
                               'ParameterApplyStatus': 'in-sync'}],
        'DBSecurityGroups': [],
        'DBSubnetGroup': {'DBSubnetGroupDescription': 'D8A Subnets',
                          'DBSubnetGroupName': 'd8a-cohesion-subnet-group',
                          'SubnetGroupStatus': 'Complete',
                          'Subnets': [{'SubnetAvailabilityZone': {'Name': 'us-west-2c'},
                                       'SubnetIdentifier': 'subnet-cbd54a92',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2a'},
                                       'SubnetIdentifier': 'subnet-b1af88d4',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2b'},
                                       'SubnetIdentifier': 'subnet-1d76336a',
                                       'SubnetStatus': 'Active'}],
                          'VpcId': 'vpc-230f5e46'},
        'DbInstancePort': 0,
        'DbiResourceId': 'db-C6RFQ3AIB6WG2IGYW6KFFY5Q5M',
        'DomainMemberships': [],
        'Engine': 'aurora',
        'EngineVersion': '5.6.10a',
        'LicenseModel': 'general-public-license',
        'MasterUsername': 'cohesion',
        'MonitoringInterval': 0,
        'MultiAZ': False,
        'OptionGroupMemberships': [{'OptionGroupName': 'default:aurora-5-6',
                                    'Status': 'in-sync'}],
        'PendingModifiedValues': {},
        'PreferredBackupWindow': '07:00-09:00',
        'PreferredMaintenanceWindow': 'thu:10:40-thu:11:10',
        'PromotionTier': 1,
        'PubliclyAccessible': False,
        'ReadReplicaDBInstanceIdentifiers': [],
        'StorageEncrypted': False,
        'StorageType': 'aurora',
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-c06b03a4'}]},
        'ResponseMetadata': {'HTTPStatusCode': 200,
                             'RequestId': '8eb59d05-cc77-11e6-b764-4d4d51731ce0'}}


CLUSTER_READER_INSTANCE_STATUS = {
    'DBInstances': [{
        'AllocatedStorage': 1,
        'AutoMinorVersionUpgrade': True,
        'AvailabilityZone': 'us-west-2b',
        'BackupRetentionPeriod': 1,
        'CACertificateIdentifier': 'rds-ca-2015',
        'CopyTagsToSnapshot': False,
        'DBClusterIdentifier': 'cohesion-chat-test',
        'DBInstanceClass': 'db.r3.8xlarge',
        'DBInstanceIdentifier': 'cohesion-chat-test-reader',
        'DBInstanceStatus': 'available',
        'DBName': 'cohesion',
        'DBParameterGroups': [{'DBParameterGroupName': 'default.aurora5.6',
                                'ParameterApplyStatus': 'in-sync'}],
        'DBSecurityGroups': [],
        'DBSubnetGroup': {'DBSubnetGroupDescription': 'D8A Subnets',
                          'DBSubnetGroupName': 'd8a-cohesion-subnet-group',
                          'SubnetGroupStatus': 'Complete',
                          'Subnets': [{'SubnetAvailabilityZone': {'Name': 'us-west-2c'},
                                       'SubnetIdentifier': 'subnet-cbd54a92',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2a'},
                                       'SubnetIdentifier': 'subnet-b1af88d4',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2b'},
                                       'SubnetIdentifier': 'subnet-1d76336a',
                                       'SubnetStatus': 'Active'}],
                           'VpcId': 'vpc-230f5e46'},
        'DbInstancePort': 0,
        'DbiResourceId': 'db-C6RFQ3AIB6WG2IGYW6KFFY5Q5M',
        'DomainMemberships': [],
        'Endpoint': {'Address': 'cohesion-chat-test-reader.cifgffw7w2ar.us-west-2.rds.amazonaws.com',
                     'HostedZoneId': 'Z1PVIF0B656C1W',
                     'Port': 3306},
        'Engine': 'aurora',
        'EngineVersion': '5.6.10a',
        'InstanceCreateTime': datetime(2016, 12, 27, 21, 8, 4, 86000, tzinfo=tzutc()),
        'LicenseModel': 'general-public-license',
        'MasterUsername': 'cohesion',
        'MonitoringInterval': 0,
        'MultiAZ': False,
        'OptionGroupMemberships': [{'OptionGroupName': 'default:aurora-5-6',
                                    'Status': 'in-sync'}],
        'PendingModifiedValues': {},
        'PreferredBackupWindow': '07:00-09:00',
        'PreferredMaintenanceWindow': 'thu:10:40-thu:11:10',
        'PromotionTier': 1,
        'PubliclyAccessible': False,
        'ReadReplicaDBInstanceIdentifiers': [],
        'StorageEncrypted': False,
        'StorageType': 'aurora',
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-c06b03a4'}]}],
        'ResponseMetadata': {'HTTPStatusCode': 200,
                             'RequestId': 'ecadbed9-cc78-11e6-91ff-4951e6e80f35'}}

DELETE_CLUSTER_READER = {
    'DBInstance': {
        'AllocatedStorage': 1,
        'AutoMinorVersionUpgrade': True,
        'AvailabilityZone': 'us-west-2b',
        'BackupRetentionPeriod': 1,
        'CACertificateIdentifier': 'rds-ca-2015',
        'CopyTagsToSnapshot': False,
        'DBClusterIdentifier': 'cohesion-chat-test',
        'DBInstanceClass': 'db.r3.8xlarge',
        'DBInstanceIdentifier': 'cohesion-chat-test-reader',
        'DBInstanceStatus': 'deleting',
        'DBName': 'cohesion',
        'DBParameterGroups': [{'DBParameterGroupName': 'default.aurora5.6',
                                'ParameterApplyStatus': 'in-sync'}],
        'DBSecurityGroups': [],
        'DBSubnetGroup': {'DBSubnetGroupDescription': 'D8A Subnets',
                          'DBSubnetGroupName': 'd8a-cohesion-subnet-group',
                          'SubnetGroupStatus': 'Complete',
                          'Subnets': [{'SubnetAvailabilityZone': {'Name': 'us-west-2c'},
                                       'SubnetIdentifier': 'subnet-cbd54a92',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2a'},
                                       'SubnetIdentifier': 'subnet-b1af88d4',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2b'},
                                       'SubnetIdentifier': 'subnet-1d76336a',
                                       'SubnetStatus': 'Active'}],
                          'VpcId': 'vpc-230f5e46'},
        'DbInstancePort': 0,
        'DbiResourceId': 'db-C6RFQ3AIB6WG2IGYW6KFFY5Q5M',
        'DomainMemberships': [],
        'Endpoint': {'Address': 'cohesion-chat-test-reader.cifgffw7w2ar.us-west-2.rds.amazonaws.com',
                     'HostedZoneId': 'Z1PVIF0B656C1W',
                     'Port': 3306},
        'Engine': 'aurora',
        'EngineVersion': '5.6.10a',
        'InstanceCreateTime': datetime(2016, 12, 27, 21, 8, 4, 86000, tzinfo=tzutc()),
        'LicenseModel': 'general-public-license',
        'MasterUsername': 'cohesion',
        'MonitoringInterval': 0,
        'MultiAZ': False,
        'OptionGroupMemberships': [{'OptionGroupName': 'default:aurora-5-6',
                                    'Status': 'in-sync'}],
        'PendingModifiedValues': {},
        'PreferredBackupWindow': '07:00-09:00',
        'PreferredMaintenanceWindow': 'thu:10:40-thu:11:10',
        'PromotionTier': 1,
        'PubliclyAccessible': False,
        'ReadReplicaDBInstanceIdentifiers': [],
        'StorageEncrypted': False,
        'StorageType': 'aurora',
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-c06b03a4'}]},
        'ResponseMetadata': {'HTTPStatusCode': 200,
                             'RequestId': '8512ce70-cc79-11e6-81bc-b17578752fa4'}}

DELETE_DB_CLUSTER = {
    'DBCluster': {
        'AllocatedStorage': 1,
        'AvailabilityZones': ['us-west-2a', 'us-west-2b', 'us-west-2c'],
        'BackupRetentionPeriod': 1,
        'DBClusterIdentifier': 'cohesion-chat-test',
        'DBClusterMembers': [],
        'DBClusterParameterGroup': 'default.aurora5.6',
        'DBSubnetGroup': 'd8a-cohesion-subnet-group',
        'DatabaseName': 'cohesion',
        'DbClusterResourceId': 'cluster-U2V2QLSEOKYEYEYJAEY5KUBNQI',
        'EarliestRestorableTime': datetime(2016, 12, 27, 20, 26, 43, 647000, tzinfo=tzutc()),
        'Endpoint': 'cohesion-chat-test.cluster-cifgffw7w2ar.us-west-2.rds.amazonaws.com',
        'Engine': 'aurora',
        'EngineVersion': '5.6.10a',
        'HostedZoneId': 'Z1PVIF0B656C1W',
        'LatestRestorableTime': datetime(2016, 12, 27, 21, 14, 59, 622000, tzinfo=tzutc()),
        'MasterUsername': 'cohesion',
        'Port': 3306,
        'PreferredBackupWindow': '07:00-09:00',
        'PreferredMaintenanceWindow': 'sat:10:48-sat:11:18',
        'Status': 'available',
        'StorageEncrypted': False,
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-c06b03a4'}]},
        'ResponseMetadata': {'HTTPStatusCode': 200,
                             'RequestId': '75cec49a-cc7a-11e6-a67a-991ad05f1bb2'}}

DELETE_CLUSTER_DB_INSTANCE = {
    'DBInstance': {
        'AllocatedStorage': 1,
        'AutoMinorVersionUpgrade': True,
        'AvailabilityZone': 'us-west-2b',
        'BackupRetentionPeriod': 1,
        'CACertificateIdentifier': 'rds-ca-2015',
        'CopyTagsToSnapshot': False,
        'DBClusterIdentifier': 'cohesion-chat-test',
        'DBInstanceClass': 'db.r3.8xlarge',
        'DBInstanceIdentifier': 'cohesion-chat-test-reader',
        'DBInstanceStatus': 'deleting',
        'DBName': 'cohesion',
        'DBParameterGroups': [{'DBParameterGroupName': 'default.aurora5.6',
                               'ParameterApplyStatus': 'in-sync'}],
        'DBSecurityGroups': [],
        'DBSubnetGroup': {'DBSubnetGroupDescription': 'D8A Subnets',
                          'DBSubnetGroupName': 'd8a-cohesion-subnet-group',
                          'SubnetGroupStatus': 'Complete',
                          'Subnets': [{'SubnetAvailabilityZone': {'Name': 'us-west-2c'},
                                       'SubnetIdentifier': 'subnet-cbd54a92',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2a'},
                                       'SubnetIdentifier': 'subnet-b1af88d4',
                                       'SubnetStatus': 'Active'},
                                      {'SubnetAvailabilityZone': {'Name': 'us-west-2b'},
                                       'SubnetIdentifier': 'subnet-1d76336a',
                                       'SubnetStatus': 'Active'}],
                          'VpcId': 'vpc-230f5e46'},
        'DbInstancePort': 0,
        'DbiResourceId': 'db-C6RFQ3AIB6WG2IGYW6KFFY5Q5M',
        'DomainMemberships': [],
        'Endpoint': {'Address': 'cohesion-chat-test-reader.cifgffw7w2ar.us-west-2.rds.amazonaws.com',
                     'HostedZoneId': 'Z1PVIF0B656C1W',
                     'Port': 3306},
        'Engine': 'aurora',
        'EngineVersion': '5.6.10a',
        'InstanceCreateTime': datetime(2016, 12, 27, 21, 8, 4, 86000, tzinfo=tzutc()),
        'LicenseModel': 'general-public-license',
        'MasterUsername': 'cohesion',
        'MonitoringInterval': 0,
        'MultiAZ': False,
        'OptionGroupMemberships': [{'OptionGroupName': 'default:aurora-5-6',
                                    'Status': 'in-sync'}],
        'PendingModifiedValues': {},
        'PreferredBackupWindow': '07:00-09:00',
        'PreferredMaintenanceWindow': 'thu:10:40-thu:11:10',
        'PromotionTier': 1,
        'PubliclyAccessible': False,
        'ReadReplicaDBInstanceIdentifiers': [],
        'StorageEncrypted': False,
        'StorageType': 'aurora',
        'VpcSecurityGroups': [{'Status': 'active',
                               'VpcSecurityGroupId': 'sg-c06b03a4'}]},
        'ResponseMetadata': {
            'HTTPStatusCode': 200,
            'RequestId': '8512ce70-cc79-11e6-81bc-b17578752fa4'}}


ATTACH_VOLUME = {'AttachTime': datetime(2017, 2, 3, 22, 45, 36, 112000, tzinfo=tzutc()),
                 'Device': '/dev/xvdb',
                 'InstanceId': 'i-0f2dfeedcee401a1c',
                 'ResponseMetadata': {'HTTPStatusCode': 200,
                                      'RequestId': '53df70cf-31f0-482d-912a-b867e0fcad0a'},
                 'State': 'attaching',
                 'VolumeId': 'vol-01dd23089a7477148'}

VOLUME_ATTACHMENTS = [{'AttachTime': datetime(2017, 2, 3, 22, 45, 36, tzinfo=tzutc()),
                       'DeleteOnTermination': False,
                       'Device': '/dev/xvdb',
                       'InstanceId': 'i-0f2dfeedcee401a1c',
                       'State': 'attached',
                       'VolumeId': 'vol-01dd23089a7477148'}]


def delete_cluster_instance_side_effect(**kwargs):
    cp = DELETE_CLUSTER_DB_INSTANCE.copy()
    # might not actually be a kwarg
    cp['DBInstance']['DBClusterIdentifier'] = kwargs.get(
        'DBClusterIdentifier',
        kwargs.get('DBInstanceIdentifier').replace('-reader', ''))
    cp['DBInstance']['DBInstanceIdentifier'] = kwargs.get('DBInstanceIdentifier')
    return cp


def mock_delete_db_cluster():
    def delete_side_effects(**kwargs):
        cp = DELETE_DB_CLUSTER.copy()
        cp['DBCluster']['DBClusterIdentifier'] = kwargs.get('DBClusterIdentifier')
        return cp

    mock = MagicMock()
    mock.delete_db_instance = MagicMock(side_effect=delete_cluster_instance_side_effect)
    mock.delete_db_cluster = MagicMock(side_effect=delete_side_effects)
    return mock, None


def mock_restore_db_cluster():
    def restore(**kwargs):
        cp = DB_CLUSTER_RESTORE_TO_POINT_IN_TIME.copy()
        cp['DBCluster']['DBClusterIdentifier'] = kwargs.get('DBClusterIdentifier')
        return cp

    def info(**kwargs):
        cp = DESCRIBE_DB_CLUSTERS.copy()
        cp['DBClusters'][0]['Status'] = 'available'
        cp['DBClusters'][0]['DBClusterIdentifier'] = kwargs.get('DBClusterIdentifier')
        return cp

    def modify(**kwargs):
        cp = MODIFY_DB_CLUSTER.copy()
        cp['DBCluster']['DBClusterIdentifier'] = kwargs.get('DBClusterIdentifier')
        cp['DBCluster']['BackupRetentionPeriod'] = kwargs.get('BackupRetentionPeriod')
        return cp

    def create(**kwargs):
        cp = CREATE_DB_INSTANCE.copy()
        cp['DBInstance']['DBInstanceIdentifier'] = kwargs.get('DBInstanceIdentifier')
        cp['DBInstance']['DBInstanceClass'] = kwargs.get('DBInstanceClass')
        cp['DBInstance']['DBClusterIdentifier'] = kwargs.get('DBClusterIdentifier')
        cp['DBInstance']['Engine'] = kwargs.get('Engine')
        cp['DBInstance']['DBSubnetGroup']['DBSubnetGroupName'] = kwargs.get('DBSubnetGroupName')
        return cp

    mock = MagicMock()
    mock.describe_db_clusters = MagicMock(side_effect=info)
    mock.delete_db_instance = MagicMock(side_effect=delete_cluster_instance_side_effect)
    mock.restore_db_cluster_to_point_in_time = MagicMock(side_effect=restore)

    mock.modify_db_cluster = MagicMock(side_effect=modify)
    mock.create_db_instance = MagicMock(side_effect=create)
    return mock, None


def mock_describe_db_clusters():

    def side_effects(**kwargs):
        cp = DESCRIBE_DB_CLUSTERS.copy()
        cp['DBClusters'][0]['DBClusterIdentifier'] = kwargs.get('DBClusterIdentifier')
        partial_host = cp['DBClusters'][0]['Endpoint'].split('.')[1:]
        partial_host.insert(0, kwargs.get('DBClusterIdentifier'))
        cp['DBClusters'][0]['Endpoint'] = '.'.join(partial_host)
        return cp

    mock = MagicMock()

    mock.describe_db_clusters = MagicMock(side_effect=side_effects)
    return mock, None
