from unittest.mock import patch
from dateutil import parser as dateparser
from zenyatta.db.wal import parse_wal_backup, pick_date_from_backups, wal_backup_command

BACKUP_LIST = (b'name\tlast_modified\texpanded_size_bytes\twal_segment_backup_start\twal_segment_offset_ba'
               b'ckup_start\twal_segment_backup_stop\twal_segment_offset_backup_stop\r\nbase_0000000300008'
               b'22A0000004B_09537112\t2016-12-15T14:11:10.000Z\t\t000000030000822A0000004B\t09537112\t\t'
               b'\r\nbase_0000000300008235000000DD_11846936\t2016-12-16T14:10:18.000Z\t\t00000003000082350'
               b'00000DD\t11846936\t\t\r\nbase_0000000300008242000000DC_08164872\t2016-12-17T14:08:49.000Z'
               b'\t\t0000000300008242000000DC\t08164872\t\t\r\nbase_000000030000825000000070_03215704\t201'
               b'6-12-18T14:08:51.000Z\t\t000000030000825000000070\t03215704\t\t\r\nbase_000000030000825D0'
               b'00000EC_12082992\t2016-12-19T14:07:07.000Z\t\t000000030000825D000000EC\t12082992\t\t\r\nb'
               b'ase_000000030000826A00000026_00348064\t2016-12-20T14:07:49.000Z\t\t000000030000826A000000'
               b'26\t00348064\t\t\r\nbase_00000003000082760000007B_00000040\t2016-12-21T14:06:47.000Z\t\t0'
               b'0000003000082760000007B\t00000040\t\t\r\nbase_00000003000082830000001C_02252240\t2016-12-'
               b'22T14:12:01.000Z\t\t00000003000082830000001C\t02252240\t\t\r\nbase_000000030000828F000000'
               b'E0_05085376\t2016-12-23T14:11:54.000Z\t\t000000030000828F000000E0\t05085376\t\t\r\nbase_0'
               b'00000030000829C000000FF_16574320\t2016-12-24T14:11:29.000Z\t\t000000030000829C000000FF\t1'
               b'6574320\t\t\r\nbase_00000003000082A800000051_06261952\t2016-12-25T14:10:21.000Z\t\t000000'
               b'03000082A800000051\t06261952\t\t\r\nbase_00000003000082B4000000F2_12992024\t2016-12-26T14'
               b':10:09.000Z\t\t00000003000082B4000000F2\t12992024\t\t\r\nbase_00000003000082C2000000E1_04'
               b'182808\t2016-12-27T14:10:24.000Z\t\t00000003000082C2000000E1\t04182808\t\t\r\nbase_000000'
               b'03000082D10000006B_10307456\t2016-12-28T14:09:01.000Z\t\t00000003000082D10000006B\t103074'
               b'56\t\t\r\nbase_00000003000082DF0000009A_09613784\t2016-12-29T14:09:15.000Z\t\t00000003000'
               b'082DF0000009A\t09613784\t\t\r\n')
TASK_DATE = '20161228T120000'


def test_parse_wal_backup():
    backups = parse_wal_backup(BACKUP_LIST)
    # ensure header was removed
    assert 'name' not in backups[0]
    # ensure date we get in next test is in the backups
    assert any(item for item in backups if '2016-12-27T14:10:24.000Z' in item)


def test_pick_date_from_backups():
    task_date = dateparser.parse(TASK_DATE)
    backups = parse_wal_backup(BACKUP_LIST)
    date, backup = pick_date_from_backups(task_date, backups)
    assert date is not None
    assert backup is not None
    assert '2016-12-27T14:10:24.000Z' in next(item for item in backups if backup in item)


@patch('zenyatta.db.wal.run_command', return_value=BACKUP_LIST)
def test_wal_backup_command(run_mock):
    creds = {'access_key': 'key',
             'secret_key': 'secret',
             'region': 'us-west-2',
             'bucket': 's3://bucket',
             'schema': 'sitedb'}
    command = wal_backup_command(creds, '/mnt', TASK_DATE, {})
    assert command[0].endswith('scripts/setup-postgres-dir.sh')
    assert command[1] == '9.5'
    assert command[2] == creds['region']
    assert command[3] == creds['bucket']
    assert command[4] == "'2016-12-28T12:00:00 UTC'"
    assert command[5] == 'base_00000003000082C2000000E1_04182808'
    assert command[6] == '/mnt'
