import logging
import psutil
import socket
from time import sleep
from typing import List, Dict
from airflow.settings import conf
from zenyatta.aws import boto_client, boto_resource


def next_letter(letter):
    int_value = ord(letter)
    assert 97 <= int_value <= 122  # assert between a-z
    if int_value == 122:  # this is z
        return 'c'  # a and b are always in use by the host
    else:
        new_char = int_value + 1
        if new_char < 99:
            return 'c'
        else:
            return chr(new_char)


def existing_devices() -> List[str]:
    return sorted([p.device if len(p.device) == 9 else p.device[:-1] for p in psutil.disk_partitions()])


def get_free_device_from_self(mapped_devices):
    default_device = '/dev/xvdc'
    devices = existing_devices()
    # devices[-1] is the last device in list and is therefore a previous device
    if devices:
        last_device = devices[-1]
        free_device = '/dev/xvd' + next_letter(last_device[-1])
        counter = 0
        # a and be are already taken, only 24 letters available
        while free_device in mapped_devices and counter < 24:
            free_device = '/dev/xvd' + next_letter(free_device[-1])
            counter += 1
        return free_device
    else:
        return default_device


def get_free_device_from_block_mapings(block_device_mappings):
    base_case = '/dev/xvdc'
    devices = sorted([d['DeviceName'] for d in block_device_mappings])
    if base_case not in devices:
        return base_case
    else:
        # return a root string + the next letter following the last letter of the last device in devices
        return '/dev/xvd' + next_letter(devices[-1][-1])


class EBSVolume:

    def __init__(self, identifier=None, size=None, iops=None, volume_type=None,
                 availability_zone=None, role_arn: str=None):
        # validate
        assert availability_zone is not None
        assert volume_type is not None and volume_type in ['gp2', 'io1']
        if volume_type is 'io1':
            assert iops is not None

        self.identifier = identifier
        self.iops = iops
        self.size = size

        self.volume_type = volume_type
        self.availability_zone = availability_zone
        self.role_arn = role_arn

    def _boto(self):
        client, _ = boto_client('ec2', self.role_arn)
        return client

    def _boto_resource(self):
        resource, _ = boto_resource('ec2', self.role_arn)
        return resource

    def create(self):
        op_kwargs = {'Size': int(self.size),
                     'VolumeType': self.volume_type,
                     'AvailabilityZone': self.availability_zone}
        if self.iops:
            op_kwargs.update({'Iops': self.iops})

        client = self._boto()
        ebs = client.create_volume(**op_kwargs)

        self.identifier = ebs['VolumeId']
        # tag volume for easy clean up
        client.create_tags(Resources=[self.identifier], Tags=[{'Key': 'name', 'Value': 'zenyatta-ebs'},
                                                              {'Key': 'Owner', 'Value': 'd8a@twitch.tv'}])
        return ebs

    def destroy(self):
        ebs = self._boto_resource().Volume(self.identifier)
        try:  # just make sure we're not attached first
            resp = ebs.detach_from_instance(Force=True)
        except:
            pass  # who cares

        ebs.delete()
        return

    def attach(self, instance_id, device_id):
        client = self._boto()
        response = client.attach_volume(VolumeId=self.identifier,
                                        InstanceId=instance_id,
                                        Device=device_id)
        return response

    def detach(self, instance_id, device):
        ebs = self._boto_resource().Volume(self.identifier)
        return ebs.detach_from_instance(InstanceId=instance_id, Device=device, Force=True)

    def status(self):
        ebs = self._boto_resource().Volume(self.identifier)
        return ebs.state

    def attach_status(self, instance_id):
        ec2 = self._boto_resource()
        volume = ec2.Volume(self.identifier)
        for attachment in volume.attachments:
            if instance_id in attachment['InstanceId']:
                return attachment['State']

        return 'detached'

    def available(self):
        status = self.status()
        if status == 'creating' or status == 'in-use':
            return False
        elif status == 'available':
            return True
        else:
            raise ValueError("ebs volume in an unexpected state: {status}".format(**locals()))

    @staticmethod
    def volume_from_id(volume_id: str, role_arn: str):
        ec2, _ = boto_resource('ec2', role_arn)
        volume = ec2.Volume(volume_id)
        return EBSVolume(identifier=volume_id,
                         availability_zone=volume.availability_zone,
                         volume_type=volume.volume_type,
                         role_arn=role_arn)


class EC2Instance:

    def __init__(self, name, identifier=None, image_id='',
                 instance_type=None,
                 security_groups=None,
                 subnet_id=None,
                 private_ip_address=None,
                 role_arn: str=None):

        self.ebs_optimized = True
        self.image_id = image_id
        self.identifier = identifier
        self.name = name
        self.instance_type = instance_type
        self.security_groups = security_groups
        self.subnet_id = subnet_id
        self.private_ip_address = private_ip_address

        self.role_arn = role_arn

        self.used_devices = []

    def _boto_client(self):
        client, _ = boto_client('ec2', self.role_arn)
        return client

    def _boto_resource(self):
        resource, _ = boto_resource('ec2', self.role_arn)
        return resource

    def attach_volume(self, volume: EBSVolume, device: str, attach_attempts: int=0):
        ec2 = self._boto_resource()
        instance = ec2.Instance(self.identifier)
        response = instance.attach_volume(VolumeId=volume.identifier,
                                          Device=device)
        wait_attempts = 0
        while volume.attach_status(self.identifier) != 'attached' and wait_attempts < 10:
            status = volume.attach_status(self.identifier)
            logging.info("waiting for {volume.identifier}:{status} on {self.identifier}".format(**locals()))
            wait_attempts += 1
            sleep(60)

        if volume.attach_status(self.identifier) != 'attached' and attach_attempts <= 5:
            self.detach_volume(volume, device, force=True)
            self.attach_volume(volume, device, attach_attempts=attach_attempts+1)

        if attach_attempts > 5:
            # task will fail here, so let's just kill this volume
            raise SystemError("could not attach volume: {} as device: {} to ec2: {}"
                              .format(volume.identifier, device, self.identifier))

        return response

    def detach_volume(self, volume: EBSVolume, device: str, force: bool=False):
        client = self._boto_client()
        response = client.detach_volume(VolumeId=volume.identifier,
                                        InstanceId=self.identifier,
                                        Device=device,
                                        Force=force)
        try:
            while volume.attach_status(self.identifier) != 'detached':
                status = volume.attach_status(self.identifier)
                logging.info("waiting for {volume.identifier}:{status} on {self.identifier}"
                             .format(**locals()))
                sleep(15)
        except ValueError as ve:
            # return current status
            if volume.status() == 'available':
                return volume.status()
            else:
                raise

        return volume.status()

    def availability_zone(self):
        ec2 = self._boto_resource()
        instance = ec2.Instance(self.identifier)
        subnet = ec2.Subnet(instance.subnet_id)
        return subnet.availability_zone

    def sunbets_zone(self, tags_dict: Dict):
        ec2 = self._boto_resource()
        instance = ec2.Instance(self.identifier)
        subnets = [subnet for subnet in instance.vpc.subnets.all()
                   for tag in subnet.tags
                   if tag['Key'] == tags_dict['key'] and tags_dict['value'] in tag['Value']]
        return dict((sub.availability_zone, sub.id) for sub in subnets)

    @staticmethod
    def instance_from_self(role_arn: str):
        name, instance_id = get_name_and_instance_id_from_hostname(socket.gethostname())
        return EC2Instance.instance_from_id(instance_id, role_arn)

    @staticmethod
    def instance_from_id(instance_id: str, role_arn: str):
        ec2, _ = boto_resource('ec2', role_arn)
        instance = ec2.Instance(instance_id)
        name = next(t['Value'] for t in instance.tags if t['Key'] == 'Name')
        logging.info("determined name: {name} and response of lookup: {instance}".format(**locals()))
        return EC2Instance(identifier=instance.instance_id,
                           image_id=instance.image_id,
                           name=name,
                           instance_type=instance.instance_type,
                           security_groups=instance.security_groups,
                           subnet_id=instance.subnet_id,
                           private_ip_address=instance.private_ip_address,
                           role_arn=role_arn)

    @staticmethod
    def get_block_devices_from_id(instance_id: str, role_arn: str):
        ec2, _ = boto_resource('ec2', role_arn)
        instance = ec2.Instance(instance_id)
        return [device['DeviceName'] for device in instance.block_device_mappings]


def get_name_and_instance_id_from_hostname(hostname):
    name = '-'.join(hostname.split('-')[:-1])
    instance_id = 'i-' + hostname.split('-')[-1]
    return name, instance_id


def create_ec2_instance(image_id: str=None,
                        instance_type: str=None,
                        security_groups: List[dict]=None,
                        min_count: int=1,
                        max_count: int=1,
                        subnet_id: str=None,
                        availability_zone: str=None,
                        role_arn: str=None,
                        **kwargs):
    ec2, _ = boto_resource('ec2', role_arn)

    instances = ec2.create_instances(ImageId=image_id,
                                     MinCount=min_count,
                                     MaxCount=max_count,
                                     SecurityGroupIds=[g['GroupId'] for g in security_groups],
                                     SubnetId=subnet_id,
                                     InstanceType=instance_type,
                                     Placement={'AvailabilityZone': availability_zone},
                                     IamInstanceProfile={'Arn': conf.get('aws', 'profile_arn')})
    # wait for them to be ready
    for instance in instances:
        instance.wait_until_running()

    # tag them so we can easily find and classify them
    task_instance = kwargs.get('task_instance')
    for instance in instances:
        instance.create_tags(Tags=[
            {'Key': 'Name',
             'Value': 'zenyatta-' + task_instance.dag_id + '-' + kwargs.get('ts_nodash')},
            {'Key': 'Owner', 'Value': 'd8a@twitch.tv'}])
    # forward VolumeId to next task
    key = task_instance.dag_id + '-instance_id-'+kwargs.get('ts_nodash')
    instance_ids = [instance.instance_id for instance in instances]
    task_instance.xcom_push(key=key, value=instance_ids)
    # return for testing
    return instance_ids
