import logging
from time import sleep
from typing import Dict, Any, List
from zenyatta.aws import get_aws_metadata, boto_client
from botocore.exceptions import ClientError
import zenyatta


class ElasticCache():

    def __init__(self, resource: str):
        meta = get_aws_metadata(resource)
        self.client = boto_client(resource, None, meta.get('region', None))[0]

    def fetch_cache_clusters(self, cache_cluster_id: str) -> List[Any]:
        cc_key = 'CacheClusters'
        retry_attampt = 3
        cc = self.client.describe_cache_clusters(ShowCacheNodeInfo=True)
        while cc_key not in cc.keys() and retry_attampt > 0:
            sleep(5)
            cc = self.client.describe_cache_clusters(ShowCacheNodeInfo=True)
            retry_attampt -= 1
        if retry_attampt == 0:
            raise Exception('fetch_cache_clusters timeout: {ResponseMetadata}'.format(**cc))
        else:
            airflow_cc_list = filter(lambda x: x['Engine'] == 'redis' and
                                     x['CacheClusterId'] == cache_cluster_id,
                                     [item for item in cc[cc_key]])
            return list(airflow_cc_list)

    def create_redis_cluster(self, cfg: dict, cache_cluster_id: str, cache_cluster_tag: str,
                             subnet_group: str, security_group_list: List[str]) -> Dict:
        try:
            tags = []
            tag = {}
            tag['Key'] = 'buddy-team'
            tag['Value'] = cache_cluster_tag
            tags.append(tag)
            response = self.client.create_cache_cluster(
                CacheClusterId=cache_cluster_id,
                NumCacheNodes=cfg['redis']['num_cache_nodes'],
                CacheNodeType=cfg['redis']['cache_node_type'],
                Engine=cfg['redis']['engine'],
                EngineVersion=cfg['redis']['engine_version'],
                CacheParameterGroupName=cfg['redis']['cache_parameter_group_name'],
                CacheSubnetGroupName=subnet_group,
                SecurityGroupIds=security_group_list,
                Tags=tags,
                PreferredMaintenanceWindow=cfg['redis']['preferred_maintenance_window'],
                Port=cfg['redis']['port'],
                AutoMinorVersionUpgrade=cfg['redis']['auto_minor_version_upgrade'],
                SnapshotRetentionLimit=cfg['redis']['snapshot_retention_limit'],
                SnapshotWindow=cfg['redis']['snapshot_window']
            )
            if response['ResponseMetadata']['HTTPStatusCode'] != 200:
                logging.info(response['ResponseMetadata'])
                raise zenyatta.common.errors.ZenyattaError(
                        "failed to delete snapshot {}".format(cache_cluster_id))
            else:
                return response
        except Exception as e:
            logging.info("failed to create redis cluster: {}".format(e))

    def delete_cache_cluster(self, cache_cluster_id: str) -> Dict:
        try:
            response = self.client.delete_cache_cluster(
                CacheClusterId=cache_cluster_id,
                FinalSnapshotIdentifier=cache_cluster_id+'-snapshot'
            )
            if response['ResponseMetadata']['HTTPStatusCode'] != 200:
                raise Exception('failed to delete credis cluster %s! %s'
                                % (cache_cluster_id, response['ResponseMetadata']))
            while len(self.fetch_cache_clusters(cache_cluster_id)) != 0:
                sleep(15)
            response = self.client.delete_snapshot(
                SnapshotName=cache_cluster_id+'-snapshot'
            )
            if response['ResponseMetadata']['HTTPStatusCode'] != 200:
                logging.info(response['ResponseMetadata'])
                raise zenyatta.common.errors.ZenyattaError(
                        "failed to delete snapshot {}".format(cache_cluster_id))
            return response
        except Exception as e:
            logging.info("failed to delete redis cluster: {}".format(e))
