/*
Package cors provides funcitonality and middleware for allowing and restricting
cross-origin requests. See also: https://www.w3.org/TR/cors/

Protocol Overview

CORS is commonly misinterpreted as a serverside form of authentication,
where the server takes the Origin header, and rejects access to the resouce
with CORS headers based on its interpretation of internal rules.

In actuality, CORS represents a series of headers sent to the client to allow it to
make decisions about whether it can allow the an XHR object to access the resource. This
package tries to express these headers using an abstraction as thin as possible.

When you call Middleware, or MustMiddleware on an http.Handler, each field in the
Policy struct is coerced into a string form and put into an http.Header via the Headers
method, where the header names are the struct tags.
These headers are appended to any response that passes through the handler.

Preflight Request Protocol

Another important thing to note is that modern CORS clients perform what's called a
'preflight request', in which they send an OPTIONS request to the endpoint to determine
its CORS rules before the real request is made -- if it's not allowed to make the
real request, it won't. These results are cached by the client for a time
determined by Policy's MaxAge field (the Access-Control-Max-Age header).

So that your program's functionality isn't still executed when a preflight requests happens in
CORS, this package provides `BlockOnOptions`, which is a middleware wrapping an `http.Request`,
preventing it from executing if the `http.Request` is an OPTIONS request. Be sure to set
this as one of your last, if not your last Middleware, so that important middleware
functionality like rate limiting and compression is not missed out when an OPTIONS
request is sent, as it will end your middleware chain.

Using this Package

The Policy struct is the most important part of this package, start there.
The example provided is essentially this:
	CORSPolicy = cors.Policy {
		AllowedOrigins: cors.Origins("localhost", "cool.com"),

		MaxAge: cors.MaxAge(5 * time.Hour),
	}

	func ContentHandler(w http.ResponseWriter, r *http.Request) {
		w.Write([]byte("hello"))
	}

	//if OPTIONS happens, we block out the ContentHandler for the preflight request, so
	//we don't perform an action / send data
	var mainHandler = CORSPolicy.MustMiddleware(
		cors.BlockOnOptions(http.HandlerFunc(ContentHandler)))
If all goes well, you should get:
	OPTIONS / HTTP/1.1
	Host: some host

	HTTP/1.1 200 OK
	Access-Control-Allow-Origin: localhost cool.com
	Access-Control-Max-Age: 18000
	Content-Type: text/plain; charset=utf-8
	Date: some date
	Content-Length: 0

	GET / HTTP/1.1
	Host: some host

	HTTP/1.1 200 OK
	Content-Length: 5
	Access-Control-Allow-Origin: localhost cool.com
	Access-Control-Max-Age: 18000
	Content-Type: text/plain; charset=utf-8
	Date: some date

	hello
*/
package cors

import (
	"encoding"
	"fmt"
	"net/http"
	"net/textproto"
	"reflect"
	"strings"
	"time"
)

//Policy represents a CORS policy. See the pkgdoc for more details.
type Policy struct {
	//A space separated list of origins that are allowed, or "*" or "null".
	//
	//When this field is zero, this header will not be sent, which causes the browser
	//to default to same origin policy.
	AllowedOrigins OriginList "Access-Control-Allow-Origin"

	//By default, browsers will not send credentials such as cookies,
	//client-cerificate authentication or HTTP simple authenitcation
	//with a CORS request. If this is set to "true" and withCredentials
	//is set on the XHR object, credentials can / will be sent.
	//
	//When this field is zero, this header will not be sent, which does not allow
	//credentials.
	AllowCredentials AllowCredentials "Access-Control-Allow-Credentials"

	//By default, in cross-site XMLHttpRequest invocations, browsers will
	//allow only access to the so-called "simple response headers", which are
	//Cache-Control, Content-Language, Content-Type, Expires, Last-Modified and
	//Pragma in addition to any of their case-insensitive equivilents. If this
	// header is specified, the comma-separated headers set here are effectively
	//added to this list for this response.
	//
	//When this field is zero, this header will not be sent, which does not allow
	//additional header access.
	ExposeHeaders HeaderList "Access-Control-Expose-Headers"

	//When performing certain types of cross-domain AJAX requests, modern browsers
	//that support CORS will insert an extra "preflight" request to get CORS details.
	//This field specifies how long the CORS information gleaned from such a request
	//is kept before it's requested again.
	//
	//When this field is zero, no such header will be sent, which allows the browser
	//to decide (usually zero).
	MaxAge MaxAge "Access-Control-Max-Age"

	//By default, in cross-site XMLHttpRequest invocations, browsers will not allow
	//methods other
	//than the "simple methods" to be used, which are: GET, HEAD and POST. Headers
	//specified here are allowed in addition to these.
	//
	//When this field is zero, no such header will be sent, which only allows GET,
	//HEAD and POST to be used.
	AllowMethods MethodList "Access-Control-Allow-Methods"

	//By default, in cross-site XMLHttpRequest invocations, browsers will only allow
	//"simple headers" that are not Content-Type to be fully altered in the
	//XMLHttpRequest object. These are: Cache-Control, Content-Language,
	//Content-Type, Expires, Last-Modified and Pragma in addition to any of
	//their case-insensitive equivilents. Content-Type may be modified, but only a
	//few Content-Types are initially allowed to be set. If this header is specified
	//the comma-separated headers set here are effectively added to this list for
	//this request.
	//
	//When this field is zero, no such header will be sent, which only allows the
	//"simple headers"
	//to be modified in a request. Additionally, Content-Type will have only a few
	//valid values.
	AllowHeaders HeaderList "Access-Control-Allow-Headers"
}

//MaxAge represents a time.Duration for a preflight request to cause
//CORS rules to be cached. It is converted into seconds in the header.
type MaxAge time.Duration

//Returns the number of seconds as a string.
func (m MaxAge) String() string { return fmt.Sprintf("%d", time.Duration(m)/time.Second) }

// unexported because this error should never be caught
type errNoTextValue reflect.Value

func (e errNoTextValue) Error() string {
	return fmt.Sprintf("no text value of %s %s", reflect.Value(e).Type(), reflect.Value(e))
}

//BlockOnOptions returns a handler which only proceeds to the next handler
//if the method is not OPTIONS. See package header doc for why this
//is useful.
func BlockOnOptions(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.Method == "OPTIONS" {
			return
		}

		h.ServeHTTP(w, r)
	})
}

func getTextValue(v reflect.Value) (s []byte, err error) {
	switch v := v.Interface().(type) {
	case encoding.TextMarshaler:
		return v.MarshalText()
	case fmt.Stringer:
		s = []byte(v.String())
		return
	}

	switch v.Kind() {
	case reflect.String:
		s = []byte(v.String())
		return
	}

	err = errNoTextValue(v)
	return
}

//Headers renders the Policy's corresponding CORS rule HTTP headers
//into an http.Header to be combined with a request.
func (p Policy) Headers() (h http.Header, err error) {
	h = make(http.Header)

	v := reflect.ValueOf(p)
	t := v.Type()

	// iterate over the struct fields and set them as headers in our
	// header fragment (cache)
	nf := t.NumField()
	for i := 0; i < nf; i++ {

		var key = string(t.Field(i).Tag)
		//skip if there's no header name (struct tag)
		if key == "" {
			return
		}

		var text []byte
		text, err = getTextValue(v.Field(i))
		if err != nil {
			return
		}

		//skip if there's no header value
		strText := string(text)
		if strText == "" {
			continue
		}

		h[key] = []string{strText}
	}

	return
}

//Middleware applies the CORS policy to the http.Handler h.
func (p Policy) Middleware(h http.Handler) (ho http.Handler, err error) {
	headers, err := p.Headers()
	if err != nil {
		return
	}

	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		for k, v := range headers {
			w.Header()[k] = append(w.Header()[k], v...)
		}

		h.ServeHTTP(w, r)
	}), nil
}

//MustMiddleware applies the CORS policy to the http.Handler h like .Middleware,
//but will panic if an error occurs.
func (p Policy) MustMiddleware(h http.Handler) (o http.Handler) {
	var err error
	o, err = p.Middleware(h)
	if err != nil {
		panic(err)
	}

	return
}

//HeaderList represents a comma-separated list of HTTP headers, see Headers for
//more information.
type HeaderList string

//Headers canonicalises its arguments Like-This and joins the given headers
//with ", ".
func Headers(h ...string) HeaderList {
	for i, v := range h {
		h[i] = textproto.CanonicalMIMEHeaderKey(v)
	}

	return HeaderList(strings.Join(h, ", "))
}

//MethodList represents a comma-separated list of HTTP methods, see
//Methods for more information.
type MethodList string

//Methods canonicalises its argument methods to upper case and joins with ", ".
func Methods(m ...string) MethodList {
	for i, v := range m {
		m[i] = strings.ToUpper(v)
	}

	return MethodList(strings.Join(m, ", "))
}

//AllowCredentials represents strings allowed as values for the
//Access-Control-Allow-Credentials CORS header. See the AllowCredentials
//Policy field for more information.
type AllowCredentials string

const (
	//True allows credentials to be sent as part of this CORS request.
	//see the AllowCredentials field of Policy for more information.
	True AllowCredentials = "true"
)

//An OriginList is a space-separated list of identifiers specifying which
//origins are allowed. See http://tools.ietf.org/html/rfc6454#section-7.1
//"origin-list-or-null" for a formal specification, and the AllowedOrigins
//field of Policy for more information.
type OriginList string

const (
	//Any allows this endpoint to be accessed by any origin. In tandem with True, it
	//can be very dangerous. See the AllowedOrigin field of Policy for more
	//information.
	Any OriginList = "*"
	//Null prevents this endpoint being accessed by any origin. See the
	//AllowedOrigin field of Policy for more information.
	Null OriginList = "null"
)

//Origins joins the Origins in o with spaces.
func Origins(o ...OriginList) OriginList {
	var out OriginList
	for _, v := range o {
		out += v + " "
	}

	out = out[:len(out)-1]

	return out
}
