package guardian

import (
	"log"
	"time"

	"code.justin.tv/systems/guardian/cfg"
	"code.justin.tv/systems/guardian/osin"
	awssession "github.com/aws/aws-sdk-go/aws/session"
	"github.com/satori/go.uuid"
)

// CreateTestDB returns a test db
func CreateTestDB() (storage *Storage) {
	c := cfg.DefaultConfig()
	c.DB.ClientsTable = cfg.DefaultTestClientsTable
	c.DB.AuthorizationsTable = cfg.DefaultTestAuthorizationsTable
	c.DB.AWSSession = awssession.New(c.DB.AWSConfig)
	storage = NewDynamoDBStorage(c, &TestIdentifier{})
	return
}

// CreateTestAuthorizeData generates a new auth code
func CreateTestAuthorizeData(client *Client) (authData *osin.AuthorizeData, err error) {
	authData = &osin.AuthorizeData{
		Client:      client,
		CreatedAt:   time.Now(),
		ExpiresIn:   0,
		UserData:    TestUser,
		State:       "state",
		RedirectURI: cfg.DefaultDevBaseURL,
		Scope:       "",
	}
	authData.Code, err = DefaultTokenGenerator.GenerateAuthorizeToken(authData)
	if err != nil {
		return
	}
	return
}

// CreateTestAccessData generates a new osin.AccessData. Does not persist data.
func CreateTestAccessData(client *Client) (accessData *osin.AccessData) {
	accessData = &osin.AccessData{
		Client:      client,
		RedirectURI: cfg.DefaultDevBaseURL,
		CreatedAt:   time.Now(),
		ExpiresIn:   0,
		UserData:    TestUser,
	}
	var err error
	accessData.AccessToken, _, err = DefaultTokenGenerator.GenerateAccessToken(accessData, false)
	if err != nil {
		log.Fatalln(err)
	}
	return
}

// TestClient persists a new test client, and returns the new object
func TestClient(db *Storage) *Client {

	client := NewTestClient()

	err := db.SaveClient(client)
	if err != nil {
		log.Fatal(err)
	}

	return client
}

// NewTestClient creates a new test client, does not persist data
func NewTestClient() *Client {
	uuid := uuid.NewV4()
	name := "testapp" + uuid.String()
	description := "really cool test app"
	homepage := "http://twitch.tv"
	redirect := "http://localhost:8000"

	client, newErr := NewClient(name, redirect, description, homepage)
	if newErr != nil {
		log.Fatal(newErr)
	}

	return client
}
