package guardian

import (
	"crypto/rand"
	"encoding/hex"
	"errors"

	"code.justin.tv/systems/guardian/osin"
)

// Default hash algorithm for guardian is sha256
const (
	DefaultHashAlgorithm = "sha256"
	DefaultSecretLength  = 20
)

// DefaultTokenGenerator is a global, shared instance of a token generator
var DefaultTokenGenerator = &TokenGenerator{
	Length:        DefaultSecretLength,
	HashAlgorithm: DefaultHashAlgorithm,
}

// TokenGenerator implements AccessTokenGen and AuthorizeTokenGen
type TokenGenerator struct {
	Length        int
	HashAlgorithm string
}

// GenerateSecret reads from crypto.Random and returns hexadecimal string value
func (t *TokenGenerator) GenerateSecret() (secret string, bytes []byte, err error) {
	bytes = make([]byte, t.Length)
	_, err = rand.Read(bytes)
	if err != nil {
		return
	}

	secret = hex.EncodeToString(bytes)
	return
}

// HashSecret hashes secret with configured algorithm. One of secret or secretBytes is required.
func (t *TokenGenerator) HashSecret(secret string, secretBytes []byte) (secretHash string, secretHashBytes []byte, err error) {
	if (secret == "") == (secretBytes == nil) {
		err = errors.New("TokenGenerator: one of secret or secretBytes is required")
		return
	}

	hasher, err := osin.GetHasher(t.HashAlgorithm)
	if err != nil {
		return
	}

	if secret != "" {
		secretBytes, err = hex.DecodeString(secret)
		if err != nil {
			return
		}
	}

	_, err = hasher.Write(secretBytes)
	if err != nil {
		return
	}

	secretHashBytes = hasher.Sum(nil)
	secretHash = hex.EncodeToString(secretHashBytes)
	return
}

// GenerateAuthorizeToken generates a token for temporary authorization
func (t *TokenGenerator) GenerateAuthorizeToken(data *osin.AuthorizeData) (ret string, err error) {
	ret, _, err = t.GenerateSecret()
	return
}

// GenerateAccessToken generates an access token
func (t *TokenGenerator) GenerateAccessToken(data *osin.AccessData, generaterefresh bool) (accesstoken string, refreshtoken string, err error) {
	accesstoken, _, err = t.GenerateSecret()
	if err != nil {
		return
	}

	if generaterefresh {
		refreshtoken, _, err = t.GenerateSecret()
		if err != nil {
			return
		}
	}

	data.Version = data.Client.GetVersion()
	return
}
