package errors

import (
	"fmt"
	"net/http"

	"code.justin.tv/systems/guardian/guardian"
)

// ForbiddenError is returned when attempting to create a policy containing
// keys and/or namespaces the user does not have access to view.
type ForbiddenError struct {
	Keys []string       `json:"keys"`
	User *guardian.User `json:"user"`
}

func (fe ForbiddenError) String() string { return fe.Error() }

func (fe ForbiddenError) Error() string {
	return fmt.Sprintf("user %s does not have access to keys %s. Users groups: %v", fe.User.CN, fe.Keys, fe.User.Groups)
}

// Code allows for adherence to APIErrorer
func (fe ForbiddenError) Code() int { return http.StatusForbidden }

// AddKeys is a helper to add key to Keys slice
func (fe *ForbiddenError) AddKeys(keys []string) {
	fe.Keys = append(fe.Keys, keys...)
}

// IsForbiddenError checks to see if error is of type ForbiddenError
func IsForbiddenError(err error) bool {
	if _, ok := err.(*ForbiddenError); ok {
		return true
	}
	return false
}

//InvalidRequestError represents a malformed request
type InvalidRequestError struct {
	Message error
}

// Error returns the error string
func (ire InvalidRequestError) Error() string {
	return ire.Message.Error()
}

// Code is the status code associated with the error
func (ire InvalidRequestError) Code() int { return 422 }
