package main

import (
	"bufio"
	"bytes"
	"flag"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net"
	"os"
	"strconv"
	"strings"
	"time"

	"github.com/deckarep/golang-set"
	"github.com/quipo/statsd"
)

var (
	cluster  string // zk namespace prefix in stats
	server   string // host:port
	interval int    // get stats from

	host      string     //zk host to connect to host:port
	staticSet mapset.Set // static set for what mntr returned
	vpcid     string     //vpc id of the zk
)

func sendStats(inStats string, value int64) {

	var clusterBuffer bytes.Buffer

	clusterBuffer.WriteString(cluster)
	clusterBuffer.WriteString(".")

	prefix := clusterBuffer.String()
	statsdClient := statsd.NewStatsdClient(server, prefix)
	statsdClient.CreateSocket()

	defer statsdClient.Close()

	name, err := os.Hostname()
	if err != nil {
		// handle error cases
		log.Println(err)
		return
	}
	//
	var buffer bytes.Buffer

	// String concatenation
	buffer.WriteString(vpcid)
	buffer.WriteString(".")
	buffer.WriteString(strings.Replace(name, ".", "_", -1))
	buffer.WriteString(".")
	buffer.WriteString(inStats)
	fmt.Println("Send: ", buffer.String(), " value: ", value)
	statsdClient.Gauge(buffer.String(), value)

}

func parseInline(inStatsMap map[string]int64) {

	//iterate through the static set for all the stats
	for setMember := range staticSet.Iter() {
		if val, ok := inStatsMap[setMember.(string)]; ok {
			sendStats(setMember.(string), val)
		} else {
			sendStats(setMember.(string), 0)
		}
	}
}

func readStats(reader *bufio.Reader) map[string]int64 {
	// declare a map for put the stats
	statsMap := make(map[string]int64)

	for {
		status, err := reader.ReadString('\n')
		if err == io.EOF {
			break
		}

		//fmt.Print("Message from server: " + status)

		result := strings.Fields(status)
		//fmt.Println(len(result))
		//fmt.Println("key: ",result[0]," value: ",result[1]," inTheSet: ",staticSet.Contains(result[0]));

		//convert stats to integer
		if strings.EqualFold("zk_server_state", result[0]) {
			if strings.EqualFold("leader", result[1]) {
				//sending 0 as follower and 1 as leader
				statsMap[result[0]] = 1
			} else {
				statsMap[result[0]] = 0
			}
		} else {
			i, err := strconv.ParseInt(result[1], 10, 64)
			if err != nil {
				i = 0
			}
			statsMap[result[0]] = i
		}
	}

	return statsMap
}

func init() {
	// cluster name space in statsd
	flag.StringVar(&cluster, "cluster", "zk_prod", "cluster name to log under in statsd ex. zk_dev")
	// statsd servers to send to
	flag.StringVar(&server, "server", "graphite-37c291f2.dev.us-west2.justin.tv:8125", "statsd server to send to ex. graphite-test:8125")
	// number of millisecond between fetching stats
	flag.IntVar(&interval, "interval", 2000, "number of milliseconds between fetching stats.")
	// which zk server to get the stats
	flag.StringVar(&host, "host", "localhost:2181", "zk host to connect to with port number ex. localhost:2181")

	// initialization of the set
	staticSet = mapset.NewSet()
	staticSet.Add("zk_packets_sent")
	staticSet.Add("zk_avg_latency")
	staticSet.Add("zk_max_latency")
	staticSet.Add("zk_min_latency")
	staticSet.Add("zk_packets_received")
	staticSet.Add("zk_packets_sent")
	staticSet.Add("zk_num_alive_connections")
	staticSet.Add("zk_outstanding_requests")
	staticSet.Add("zk_server_state")
	staticSet.Add("zk_znode_count")
	staticSet.Add("zk_watch_count")
	staticSet.Add("zk_ephemerals_count")
	staticSet.Add("zk_approximate_data_size")
	staticSet.Add("zk_open_file_descriptor_count")
	staticSet.Add("zk_max_file_descriptor_count")

	content, err := ioutil.ReadFile("/etc/facter/facts.d/vpcid.txt")
	if err != nil {
		vpcid = "vpc-local"
		log.Println("Can not find /etc/facter/facts.d/vpcid.txt file -- default to vpc-local")
	} else {
		vpcidString := strings.TrimSpace(string(content))
		s := strings.Split(vpcidString, "=")
		if len(s) < 2 {
			vpcid = "vpc-local"
			log.Println("Can not parse content of /etc/facter/facts.d/vpcid.txt -- default to vpc-local")
		} else {
			vpcid = s[1]
		}
	}
}

func main() {
	// parse the command-line arguments
	flag.Parse()

	// started infinite loop to send the stats to statsd
	for {
		conn, err := net.Dial("tcp", host)
		if err != nil {
			//print error
			log.Println(err)
			return
		}

		defer conn.Close()

		fmt.Fprintf(conn, "mntr")

		tco := bufio.NewReader(conn)

		// reading all stats from the connection
		statsMap := readStats(tco)

		// parse the stats and send
		parseInline(statsMap)

		// amount of time to sleep
		time.Sleep(time.Duration(interval) * time.Millisecond)
	}

}
