package main

import (
	"context"
	"fmt"
	"net"
	"net/http"
	"net/url"
	"os"
	"time"

	"code.justin.tv/danielnf/destiny/destinytwirp"
	destiny "code.justin.tv/danielnf/destiny/internal"
	destinyhttp "code.justin.tv/danielnf/destiny/internal/http"
	"code.justin.tv/danielnf/destiny/internal/local"
	"code.justin.tv/danielnf/destiny/internal/postgres"
	"code.justin.tv/danielnf/destiny/internal/sqs"
	"code.justin.tv/danielnf/destiny/internal/tests"
	"code.justin.tv/danielnf/destiny/internal/twirp"
	"github.com/segmentio/conf"
	"github.com/segmentio/events"
	_ "github.com/segmentio/events/ecslogs"
	_ "github.com/segmentio/events/text"
	"github.com/segmentio/stats"
	"github.com/segmentio/stats/datadog"
	"github.com/segmentio/stats/httpstats"
	"github.com/segmentio/stats/procstats"
)

// Version is an injected variable so that it's set as
// the git commit sha of the current deployment.
var Version = "0.1.0"

type httpConfig struct {
	MaxIdleConn int           `conf:"max-idle-conn"`
	Timeout     time.Duration `conf:"timeout"`
}

type routeConfig struct {
	Subscriptions string `conf:"subscriptions"`
}

type engineConfig struct {
	ReadInterval time.Duration `conf:"read-interval"`
	MaxReaders   int           `conf:"max-readers"`
}

type config struct {
	Bind             string        `conf:"bind"`
	Version          bool          `conf:"version"`
	Region           string        `conf:"region"`
	Environment      string        `conf:"environment"`
	Debug            bool          `conf:"debug"`
	DatadogAddr      string        `conf:"datadog-addr"`
	DatadogAPMAddr   string        `conf:"datadog-apm-addr"`
	EventDB          string        `conf:"event-db"`
	Routes           routeConfig   `conf:"routes"`
	Engine           engineConfig  `conf:"engine"`
	HTTP             httpConfig    `conf:"http"`
	LoadTestRate     int           `conf:"load-test-rate"`
	LoadTestDuration time.Duration `conf:"load-test-duration"`
	LoadTestURL      string        `conf:"load-test-url"`
}

func main() {
	config := config{
		Region:      "us-west-2",
		Bind:        ":3000",
		Environment: "development",
		EventDB:     "postgres://destiny:destiny@localhost:5432/destiny?sslmode=disable",
		HTTP: httpConfig{
			Timeout:     time.Second,
			MaxIdleConn: 1000,
		},
		Routes: routeConfig{
			Subscriptions: "http://subscriptions-stag.internal.justin.tv/twirp/code.justin.tv.revenue.subscriptions.Subscriptions",
		},
		Engine: engineConfig{
			ReadInterval: 5 * time.Second,
			MaxReaders:   1,
		},
	}

	cmd, args := conf.LoadWith(nil, conf.Loader{
		Name: "destiny",
		Args: os.Args[1:],
		Commands: []conf.Command{
			{"destiny", "Run the Destiny service"},
			{"load-test", "Run the Destiny load test"},
		},
	})

	ctx, cancel := events.WithSignals(context.Background(), os.Interrupt, os.Kill)
	defer cancel()

	switch cmd {
	case "destiny":
		runDestiny(ctx, config, args)
	case "load-test":
		runLoadTest(ctx, config, args)
	}

	<-ctx.Done()

	events.Log("destiny shutting down")
}

func runLoadTest(ctx context.Context, config config, args []string) {
	conf.LoadWith(&config, conf.Loader{
		Name: "load-test",
		Args: args,
	})

	events.Log("starting load test")

	if err := tests.LoadTestSend(config.LoadTestURL, config.LoadTestRate, config.LoadTestDuration); err != nil {
		fmt.Println(err)
		os.Exit(1)
	}

	events.Log("load testing complete")
	os.Exit(0)
}

func runDestiny(ctx context.Context, config config, args []string) {
	conf.LoadWith(&config, conf.Loader{
		Name: "destiny",
		Args: args,
	})

	events.DefaultLogger.EnableDebug = config.Debug
	events.DefaultLogger.EnableSource = config.Debug

	if config.Version {
		fmt.Println("Destiny Version: ", Version)
		return
	}

	events.Log("starting destiny version => %{version}s", Version)

	makeStats(ctx, config)
	eventDB := makeEventDB(ctx, config)

	client := &http.Client{
		Transport: &http.Transport{
			Proxy: http.ProxyFromEnvironment,
			DialContext: (&net.Dialer{
				Timeout:   10 * time.Second,
				KeepAlive: 1 * time.Minute,
				DualStack: true,
			}).DialContext,
			MaxIdleConns:          config.HTTP.MaxIdleConn,
			IdleConnTimeout:       90 * time.Second,
			TLSHandshakeTimeout:   10 * time.Second,
			ExpectContinueTimeout: 1 * time.Second,
		},
		Timeout: time.Second,
	}

	routes := map[string]string{
		"subscriptions": config.Routes.Subscriptions,
	}

	httpDestination := makeDestination(destinyhttp.NewDestination(routes, client))

	destinations := map[string]destiny.Destination{
		"http":  httpDestination,
		"https": httpDestination,
		"local": makeDestination(local.NewDestination()),
		"sqs":   makeDestination(sqs.NewDestination(config.Region)),
	}

	engine := destiny.NewEngine(destiny.EngineConfig{
		DB:           eventDB,
		ReadInterval: config.Engine.ReadInterval,
		MaxReaders:   config.Engine.MaxReaders,
		Destinations: destinations,
		Stats:        stats.DefaultEngine.WithTags(stats.T("component", "engine")),
	})

	twirp := makeTwirp(config, eventDB)

	go func() {
		events.Log("listening on '%{port}s'", config.Bind)
		if err := http.ListenAndServe(config.Bind, twirp); err != nil {
			events.Log("%{error}s", err)
		}
	}()

	go func() {
		engine.Run(ctx)
	}()
}

func makeDestination(dest destiny.Destination) destiny.Destination {
	dest = destiny.NewDestinationLogger(dest, events.DefaultLogger)
	return dest
}

func makeTwirp(config config, eventDB destiny.EventDB) http.Handler {
	var handler http.Handler
	var dest destinytwirp.Destiny

	dest = twirp.NewDestiny(eventDB)
	handler = twirp.NewHandler(dest)
	handler = httpstats.NewHandler(handler)

	return handler
}

func makeEventDB(ctx context.Context, config config) (db destiny.EventDB) {
	parsed, err := url.Parse(config.EventDB)
	if err != nil {
		events.Log("failed to parse event db config: expected a url format")
		os.Exit(1)
	}

	switch parsed.Scheme {
	case "postgres":
		pg, err := postgres.NewEventDB(config.EventDB)
		if err != nil {
			fmt.Println(err)
			os.Exit(1)
		}

		if err = pg.Create(ctx); err != nil {
			fmt.Println(err)
			os.Exit(1)
		}

		db = pg
	default:
		events.Log("event db scheme '%{scheme}' is not supported", parsed.Scheme)
		os.Exit(1)
	}

	db = destiny.NewEventDBWithStats(db, stats.DefaultEngine.WithTags(stats.T("engiine", "postgres")))
	return
}

func makeStats(ctx context.Context, config config) {
	if config.DatadogAddr != "" {
		stats.DefaultEngine.Register(datadog.NewClient(config.DatadogAddr))

		tags := []stats.Tag{
			stats.T("version", Version),
			stats.T("region", config.Region),
			stats.T("env", config.Environment),
		}

		if os.Getenv("CANARY") == "true" {
			tags = append(tags, stats.T("canary", "true"))
		}

		stats.DefaultEngine = stats.DefaultEngine.WithTags(tags...)

		defer stats.Flush()

		go func() {
			// Start a new collector for the current process, reporting Go metrics.
			c := procstats.StartCollector(procstats.NewGoMetrics())

			// Gracefully stops stats collection.
			<-ctx.Done()
			c.Close()
		}()
	}
}
