package destiny

import (
	"context"
	"io"
	"net/url"
	"time"

	"github.com/segmentio/events"
	"github.com/segmentio/stats"
)

// Engine ...
type Engine struct {
	db           EventDB
	interval     time.Duration
	maxReaders   int
	destinations map[string]Destination
	stats        *stats.Engine
}

// EngineConfig ...
type EngineConfig struct {
	DB           EventDB
	ReadInterval time.Duration
	MaxReaders   int
	Destinations map[string]Destination
	Stats        *stats.Engine
}

// NewEngine ...
func NewEngine(config EngineConfig) *Engine {
	if config.MaxReaders == 0 {
		config.MaxReaders = 1
	}

	if config.Stats == nil {
		config.Stats = stats.NewEngine("", stats.Discard)
	}

	return &Engine{
		db:           config.DB,
		interval:     config.ReadInterval,
		destinations: config.Destinations,
		stats:        config.Stats,
		maxReaders:   config.MaxReaders,
	}
}

func (engine *Engine) process(ctx context.Context, offset time.Time) {
	events.Log("reading events <= '%{offset}s'", offset.Format(time.RFC3339))

	iter := engine.db.ReadEvents(ctx, offset, 3*time.Second)

	var event Event
	for iter.Next(&event) {
		engine.stats.Incr("operation.count", stats.T("operation", "read-event"))

		if event.MaxRetries == 0 {
			event.MaxRetries = 10
		}

		url, err := url.Parse(event.Destination)
		if err != nil {
			// TODO: add an event.Error() or something to remove the event
			// from the db and archive it as errored.
			if commitErr := event.Commit(); commitErr != nil {
				events.Log("failed to commit event: %{error}s (after original error: %{err}s)", commitErr, err)
				continue
			}
		}

		// TODO: implement ports
		destinyURL := URL{
			Scheme: url.Scheme,
			Host:   url.Host,
			Path:   url.Path,
		}

		dest, ok := engine.destinations[destinyURL.Scheme]
		// No destinations for that event was found. This may be a temporary error from a
		// bad deployment or configuration. To prevent losing errors we retry a few times with
		// long enough backoffs.
		if !ok {
			// TODO: make this configurable.
			if event.Attempts >= event.MaxRetries {
				if err := event.Commit(); err != nil {
					events.Log("failed to commit event after max attempts reached: %{error}s", err)
					continue
				}
			}

			engine.retryEvent(ctx, event)
			continue
		}

		ctx, cancel := context.WithTimeout(ctx, 1*time.Second)
		// TODO: implement cursor state saving.
		start := time.Now()
		if _, err := dest.HandleEvent(ctx, destinyURL, event); err != nil {
			cancel()

			switch {
			case IsErrNoRetry(err):
				engine.stats.Incr("operation.count", stats.T("operation", "discarded"))
				event.Commit()
			case IsErrRetryable(err):
				engine.retryEvent(ctx, event)
			}

			continue
		}

		engine.archiveEvent(ctx, event)
		cancel()

		engine.stats.Observe("operation.time", time.Now().Sub(start), stats.T("operation", "deliveries"))
		engine.stats.Incr("operation.count", stats.T("operation", "deliveries"))
	}

	if err := iter.Close(); err != nil && err != io.EOF {
		events.Log("error reading events: %{error}s", err)
	}
}

func (engine *Engine) archiveEvent(ctx context.Context, event Event) {
	if err := event.Commit(); err != nil {
		events.Log("failed to commit: %{error}s", err)
		return
	}
}

func (engine *Engine) retryEvent(ctx context.Context, event Event) {
	engine.stats.Incr("operation.count", stats.T("operation", "retry"))

	nextAttempt := time.Duration((event.Attempts + 1) * 2)
	retryAt := time.Now().Add(nextAttempt * time.Second)
	if err := event.Retry(retryAt); err != nil {
		events.Log("failed to mark event as retry: %{error}s", err)
	}
}

// Run ...
func (engine *Engine) Run(ctx context.Context) error {
	events.Log("started destiny engine with %{num}d readers per interval", engine.maxReaders)

	go engine.process(ctx, time.Now())

	ticker := time.NewTicker(engine.interval)
	defer ticker.Stop()

	for {
		select {
		case <-ctx.Done():
			return ctx.Err()
		case t := <-ticker.C:
			for i := 0; i < engine.maxReaders; i++ {
				go engine.process(ctx, t)
			}
		}
	}
}
