package twirp

import (
	"bytes"
	"context"
	"fmt"
	"io/ioutil"
	"net/http"

	destiny "code.justin.tv/danielnf/destiny/internal"
)

// Destination ...
type Destination struct {
	routes map[string]string
	client *http.Client
}

// NewDestination ...
func NewDestination(routes map[string]string, client *http.Client) *Destination {
	return &Destination{
		routes: routes,
		client: client,
	}
}

// HandleEvent ...
func (dest *Destination) HandleEvent(ctx context.Context, url destiny.URL, event destiny.Event) (cursor destiny.Cursor, err error) {
	route, ok := dest.routes[url.Host]
	if !ok {
		route = url.Scheme + url.Host
	}

	var contentType string
	contentType, err = makeContentType(event.Encoding)
	if err != nil {
		err = destiny.ErrNoRetry(err.Error())
		return
	}

	var req *http.Request
	var resp *http.Response
	var buffer = bytes.NewBuffer(event.Payload)
	var uri = route + url.Path

	req, err = http.NewRequest("POST", uri, buffer)
	if err != nil {
		return
	}

	req.Header.Add("Content-Type", contentType)

	resp, err = dest.client.Do(req.WithContext(ctx))
	if err != nil {
		err = destiny.ErrRetryable(err.Error())
		return
	}

	defer resp.Body.Close()

	var body []byte
	body, err = ioutil.ReadAll(resp.Body)
	if err != nil {
		return
	}

	if resp.StatusCode >= 400 && resp.StatusCode < 500 {
		err = destiny.ErrNoRetry(fmt.Sprintf("http %d: POST %s => %s", resp.StatusCode, uri, string(body)))
		return
	}

	if resp.StatusCode != 200 {
		err = destiny.ErrRetryable(fmt.Sprintf("http %d: POST %s => %s", resp.StatusCode, uri, string(body)))
		return
	}

	return
}

func makeContentType(encoder string) (ty string, err error) {
	switch encoder {
	case "protobuf":
		ty = "application/protobuf"
		return
	case "json":
		ty = "application/json"
		return
	}

	err = fmt.Errorf("encoder '%s' is not supported by the http/https destination", encoder)
	return
}
