package postgres

import (
	"context"
	"database/sql"
	"fmt"
	"io"
	"sync"
	"time"

	"github.com/segmentio/ksuid"

	destiny "code.justin.tv/danielnf/destiny/internal"
	pg "github.com/lib/pq"
	"github.com/pkg/errors"
)

// tx is a wrapper around an event transaction. Each transaction
type tx struct {
	*sql.Tx
	stmt   *sql.Stmt
	retry  *sql.Stmt
	ctx    context.Context
	cancel context.CancelFunc
}

// EventDB implements the `destiny.EventDB` interface that
// allows for delivering messages at a future time.
type EventDB struct {
	pg *sql.DB

	// ready controls the initialization of the prepared
	// statements. These are done lazily so that the operation
	// of creating prepared statements are context aware.
	ready   bool
	readyMu sync.RWMutex

	queueStmt   *sql.Stmt
	writeStmt   *sql.Stmt
	writeQAStmt *sql.Stmt
	findQAStmt  *sql.Stmt
}

// NewEventDB returns a new `destiny.EventDB` implementation that
// operates on a postgres database.
//
// An error is returned if connecting to postgres is unsuccessful.
func NewEventDB(url string) (*EventDB, error) {
	db, err := sql.Open("postgres", url)
	if err != nil {
		return nil, errors.Wrap(err, "failed to create event db")
	}

	db.SetMaxOpenConns(500)
	db.SetMaxIdleConns(500)

	return &EventDB{
		pg: db,
	}, nil
}

func (db *EventDB) initialize(ctx context.Context) (err error) {
	db.readyMu.RLock()
	ready := db.ready
	db.readyMu.RUnlock()

	if ready {
		return
	}

	db.readyMu.Lock()
	defer db.readyMu.Unlock()

	db.queueStmt, err = db.pg.PrepareContext(ctx, `
	delete from events
	where id = (
		select id
		from events
		where id <= $1
		order by id
		for update skip locked
		limit 1
	)	
	returning *;
	`)

	if err != nil {
		return errors.Wrap(err, "failed to create 'queue' prepared statement")
	}

	db.writeStmt, err = db.pg.PrepareContext(ctx, `
	insert into
		events (id, destination, domain, payload, attempts, cursor, max_retries, encoding)
	values
		($1, $2, $3, $4, $5, $6, $7, $8)
	on conflict (cursor)	
	do update set id = $1
	`)

	if err != nil {
		return errors.Wrap(err, "failed to create 'write' prepared statement")
	}

	db.writeQAStmt, err = db.pg.PrepareContext(ctx, `
	insert into
		qa_events (cursor, sent_at, created_at)
	values
		($1, $2, $3)
	`)

	if err != nil {
		return errors.Wrap(err, "failed to create 'writeQA' prepared statement")
	}

	db.findQAStmt, err = db.pg.PrepareContext(ctx, `
	select sent_at, created_at from qa_events where cursor = $1;
	`)

	if err != nil {
		return errors.Wrap(err, "failed to create 'findQAStmt' prepared statement")
	}

	db.ready = true

	return
}

// Create initializes the database tables and indexes required. If a resource (table/index)
// already exists the operation will be skipped and no error will be returned.
func (db *EventDB) Create(ctx context.Context) (err error) {
	// Ensure there's no prepared statements being created until after all the
	// resources are created.
	db.readyMu.Lock()
	defer db.readyMu.Unlock()

	queries := []string{
		`create table events (
			id                  bytea not null primary key, -- KSUID in bytes
			destination 		text not null, 				-- e.g., sqs://$queuename
			domain 				text not null, 				-- e.g., subscriptions
			payload            	bytea not null,
			cursor              text not null unique,
			attempts            int not null default 0,
			max_retries         int not null default 5,
			encoding            text not null,
			created_at          timestamp with time zone default current_timestamp not null
		);`,

		`create table archived (
			id                  bytea not null primary key,	-- KSUID in bytes
			destination 		text not null,				-- e.g., sqs://$queuename
			domain 				text not null,				-- e.g., subscriptions
			committed_at        timestamp with time zone default current_timestamp not null
		);`,

		`create table qa_events (
			cursor 			text not null primary key,
			sent_at  		timestamp with time zone default current_timestamp not null,
			created_at  	timestamp with time zone default current_timestamp not null
		);`,

		`create index idx_events_recent on events(id asc);`,
	}

	for _, query := range queries {
		if _, err = db.pg.ExecContext(ctx, query); err != nil {
			// Skip any duplicate resource errors so `(*EventDB).Create()` can be called
			// multiple times.
			if dbErr, ok := err.(*pg.Error); ok {
				if isDup(dbErr) {
					err = nil
					continue
				}
			}

			return
		}
	}

	return
}

// ReadQAEvent ...
func (db *EventDB) ReadQAEvent(ctx context.Context, cursor string) (event destiny.Event, err error) {
	row := db.findQAStmt.QueryRowContext(ctx, cursor)

	var sentAt time.Time
	if err = row.Scan(&sentAt, &event.CreatedAt); err != nil {
		return
	}

	return
}

// WriteQAEvent ...
func (db *EventDB) WriteQAEvent(ctx context.Context, event destiny.Event) (err error) {
	if err = db.initialize(ctx); err != nil {
		return
	}

	if _, err = db.writeQAStmt.ExecContext(ctx, event.Cursor, event.ID.Time(), event.CreatedAt); err != nil {
		return
	}

	return
}

// WriteEvents ...
func (db *EventDB) WriteEvents(ctx context.Context, events ...destiny.Event) (err error) {
	if err = db.initialize(ctx); err != nil {
		return
	}

	tx, err := db.pg.BeginTx(ctx, &sql.TxOptions{})
	if err != nil {
		return errors.Wrap(err, "failed to begin transaction")
	}

	for _, event := range events {
		_, err = tx.StmtContext(ctx, db.writeStmt).ExecContext(ctx,
			event.ID.Bytes(),
			event.Destination,
			event.Domain,
			event.Payload,
			0, // attempts
			event.Domain+"/"+event.Cursor,
			event.MaxRetries,
			event.Encoding,
		)

		if err != nil {
			tx.Rollback()
			err = errors.Wrap(err, "failed to write event")
			return
		}
	}

	return tx.Commit()
}

// ReadEvents ...
// txTimeout controls how long event transactions can remain open before
// being canceled. This prevents a reader from leaking too many transactions
// before committing them which will prevent those events from being visible
// until the transaction is rolled back or committed.
func (db *EventDB) ReadEvents(ctx context.Context, offset time.Time, txTimeout time.Duration) destiny.EventIter {
	if err := db.initialize(ctx); err != nil {
		return &queueIter{err: err}
	}

	offsetKSUID, err := ksuid.NewRandomWithTime(offset)
	if err != nil {
		return &queueIter{err: err}
	}

	return &queueIter{
		offset: offsetKSUID,
		ctx:    ctx,
		txs:    make(map[ksuid.KSUID]tx),
		makeTx: func() (tx tx, err error) {
			tx.ctx, tx.cancel = context.WithTimeout(ctx, txTimeout)
			tx.Tx, err = db.pg.BeginTx(tx.ctx, nil)
			if err != nil {
				return
			}

			tx.stmt = tx.StmtContext(tx.ctx, db.queueStmt)
			tx.retry = tx.StmtContext(tx.ctx, db.writeStmt)

			return
		},
	}
}

// Close will terminate any prepared statements and database connections.
func (db *EventDB) Close() error {
	if db.queueStmt != nil {
		db.queueStmt.Close()
	}

	if db.writeStmt != nil {
		db.writeStmt.Close()
	}

	return db.pg.Close()
}

func isDup(err *pg.Error) bool {
	if err.Code == "42P07" || err.Code == "42710" {
		return true
	}

	return false
}

type queueIter struct {
	offset ksuid.KSUID
	// err is a reference to the last error that occurred (if any).
	err error
	// ctx controls the entire iterator lifecycle.
	ctx context.Context
	// makeTx is responsible for creating a new transaction. This decouples
	// the postgres layer with the iterator layer.
	makeTx func() (tx, error)
	// txs is a map of event IDs to open transactions. This allows the reader to call `iter.Close()`
	// which will close any open transactions.
	txs map[ksuid.KSUID]tx
	// allow the event iterator to be used concurrently (e.g., event.Commit()/event.Retry() can
	// happen in a separate goroutine).
	txsMu sync.Mutex
}

// Next will try and read a new event from the transaction and set the pointer's value
// to a valid `destiny.Event`.
//
// False will be returned if an error has occurred. This error may be an EOF which means
// no more events were found for the given query.
func (iter *queueIter) Next(event *destiny.Event) bool {
	// Prevent more .Next calls from taking place if an error has already
	// occured.
	if iter.err != nil {
		return false
	}

	if iter.ctx.Err() != nil {
		iter.err = iter.ctx.Err()
		return false
	}

	// reset the pointer just in case.
	*event = destiny.Event{}

	var tx tx
	tx, iter.err = iter.makeTx()
	if iter.err != nil {
		return false
	}

	row := tx.stmt.QueryRowContext(tx.ctx, iter.offset.Bytes())
	if iter.err = row.Scan(
		&event.ID,
		&event.Destination,
		&event.Domain,
		&event.Payload,
		&event.Cursor,
		&event.Attempts,
		&event.MaxRetries,
		&event.Encoding,
		&event.CreatedAt,
	); iter.err != nil {
		tx.Rollback()
		return false
	}

	// Commit will finalize the transaction and remove the event from the table.
	event.Commit = func() error {
		// transaction can be removed from the tracking list.
		iter.txsMu.Lock()
		delete(iter.txs, event.ID)
		iter.txsMu.Unlock()

		defer tx.cancel()
		return tx.Commit()
	}

	// Retry will insert the event back into the table using the open transaction
	// (which deletes it once the transaction is committed). The new event will have
	// a new KSUID with an updated timestamp to allow retries to be further in the
	// future.
	event.Retry = func(at time.Time) (err error) {
		iter.txsMu.Lock()
		delete(iter.txs, event.ID)
		iter.txsMu.Unlock()

		// Set the event to retry at a potentially future time.
		event.ID, err = ksuid.NewRandomWithTime(at)
		if err != nil {
			// Unlikely but this will keep the original event in the table which will be retried
			// likely immediately.
			tx.Rollback()
			return
		}

		// Insert the new event into the table.
		_, err = tx.retry.ExecContext(tx.ctx,
			event.ID.Bytes(),
			event.Destination,
			event.Domain,
			event.Payload,
			// Keep increasing the attempt count so that it can discarded if it reaches
			// a maximum.
			event.Attempts+1,
			event.Cursor,
			event.MaxRetries,
			event.Encoding,
		)

		if err != nil {
			// Unlikely but this will keep the original event in the table which will be retried
			// likely immediately.
			tx.Rollback()
			return
		}

		defer tx.cancel()
		// Committing will delete the original event and make the new event visible to
		// any new transactions.
		return tx.Commit()
	}

	// Keep track of the transaction so it doesn't get leaked.
	iter.txsMu.Lock()
	iter.txs[event.ID] = tx
	iter.txsMu.Unlock()

	return true
}

// Close will return any errors that occured while trying to read events. It will also
// rollback any open transactions that were opened.
func (iter *queueIter) Close() error {
	// Avoid leaking any SQL details outside of this package. Exposing io.EOF allows
	// readers to check for it.
	if iter.err == sql.ErrNoRows {
		iter.err = io.EOF
	}

	iter.txsMu.Lock()
	defer iter.txsMu.Unlock()

	for _, tx := range iter.txs {
		if err := tx.Rollback(); err != nil {
			fmt.Println("123", err)
		}
		tx.cancel()
	}

	return iter.err
}
