package postgres

import (
	"context"
	"database/sql"
	"fmt"
	"os"
	"strings"
	"testing"

	destiny "code.justin.tv/danielnf/destiny/internal"
	"code.justin.tv/danielnf/destiny/internal/tests"
	_ "github.com/lib/pq"
	"github.com/pkg/errors"
	"github.com/segmentio/ksuid"
)

func createTestDatabase(ctx context.Context, url string) (string, error) {
	db, err := sql.Open("postgres", url)
	if err != nil {
		return "", err
	}

	name := "test" + strings.ToLower(ksuid.New().String())
	if _, err := db.ExecContext(ctx, fmt.Sprintf(`create database %s;`, name)); err != nil {
		return "", err
	}

	return name, nil
}

func MakeEventDB(ctx context.Context, events ...destiny.Event) (destiny.EventDB, func() error, error) {
	url := "postgres://destiny:destiny@localhost:5432/"

	if os.Getenv("DOCKER_TEST") != "" {
		url = "postgres://destiny:destiny@jenkins_postgres_1:5432/"
	}

	dbName, err := createTestDatabase(ctx, url+"?sslmode=disable")
	if err != nil {
		return nil, nil, errors.Wrap(err, "failed to create test database")
	}

	db, err := NewEventDB(url + dbName + "?sslmode=disable")
	if err != nil {
		return nil, nil, err
	}

	if err := db.Create(ctx); err != nil {
		return nil, nil, err
	}

	if err := db.WriteEvents(ctx, events...); err != nil {
		return nil, nil, err
	}

	return db, func() error { return db.Close() }, nil
}

func TestPostgresEventDB(t *testing.T) {
	tests.TestEventDB(t, MakeEventDB)
}

func BenchmarkPostgresEventDB(b *testing.B) {
	tests.BenchEventDB(b, MakeEventDB)
}
