package destiny

import (
	"context"
	"time"

	"github.com/segmentio/stats"
)

type eventDBStats struct {
	base  EventDB
	stats *stats.Engine
}

func NewEventDBWithStats(base EventDB, engine *stats.Engine) EventDB {
	return &eventDBStats{
		base:  base,
		stats: engine.WithTags(stats.T("component", "eventdb")),
	}
}

func (db *eventDBStats) WriteEvents(ctx context.Context, values ...Event) (err error) {
	op := stats.T("operation", "write-events")
	if err = db.base.WriteEvents(ctx, values...); err != nil {
		db.stats.Incr("error.count", op)
		return
	}

	db.stats.Add("operation.count", len(values), op)
	return
}

func (db *eventDBStats) ReadEvents(ctx context.Context, offset time.Time, timeout time.Duration) EventIter {
	return &eventIterStats{
		base:  db.base.ReadEvents(ctx, offset, timeout),
		stats: db.stats,
	}
}

func (db *eventDBStats) Close() error {
	return db.base.Close()
}

type eventIterStats struct {
	base  EventIter
	stats *stats.Engine
}

func NewEventIterStats(base EventIter, engine *stats.Engine) EventIter {
	return &eventIterStats{
		base:  base,
		stats: engine,
	}
}

func (iter *eventIterStats) Next(val *Event) (next bool) {
	next = iter.base.Next(val)
	if !next {
		return
	}

	iter.stats.Incr("operation.count", stats.T("operation", "read-event"))
	return
}

func (iter *eventIterStats) Close() error {
	return iter.base.Close()
}
