package tests

import (
	"context"
	"io"
	"testing"
	"time"

	"github.com/segmentio/ksuid"
	"github.com/stretchr/testify/assert"

	destiny "code.justin.tv/danielnf/destiny/internal"
)

// MakeEventDB ...
type MakeEventDB func(ctx context.Context, events ...destiny.Event) (destiny.EventDB, func() error, error)

func ksuidTime(date string) ksuid.KSUID {
	ts, err := time.Parse(time.RFC3339, date)
	if err != nil {
		panic(err)
	}

	id, err := ksuid.NewRandomWithTime(ts)
	if err != nil {
		panic(err)
	}

	return id
}

func makeTime(date string) time.Time {
	ts, err := time.Parse(time.RFC3339, date)
	if err != nil {
		panic(err)
	}

	return ts
}

// TestEventDB ...
func TestEventDB(t *testing.T, makeEventDB MakeEventDB) {
	tests := []struct {
		scenario string
		events   []destiny.Event
		test     func(ctx context.Context, t *testing.T, db destiny.EventDB, events ...destiny.Event)
	}{
		{
			scenario: "read the event and commit it",
			events: []destiny.Event{
				{
					ID:          ksuidTime("2019-01-01T05:05:05Z"),
					Domain:      "testing",
					Cursor:      ksuid.New().String(),
					Destination: "test://foobar",
					Payload:     []byte("foobar payload"),
				},
			},
			test: func(ctx context.Context, t *testing.T, db destiny.EventDB, events ...destiny.Event) {
				// Read any events starting a minute before the time of the first event.
				iter := db.ReadEvents(ctx, events[0].ID.Time().Add(-1*time.Minute), time.Second)

				committed := 0
				var event destiny.Event
				for iter.Next(&event) {
					if err := event.Commit(); err != nil {
						t.Fatal(err)
					}

					assert.Equal(t, event.ID.String(), events[committed].ID.String())
					assert.Equal(t, event.Domain, events[committed].Domain)
					assert.Equal(t, string(event.Payload), string(events[committed].Payload))
					assert.Equal(t, event.Destination, events[committed].Destination)

					committed++
				}

				if err := iter.Close(); err != io.EOF {
					t.Fatalf("expected no rows error: %s", err)
				}
			},
		},

		{
			scenario: "read many events and commit them",
			events: []destiny.Event{
				{
					ID:          ksuidTime("2019-01-01T05:05:05Z"),
					Domain:      "testing0",
					Cursor:      ksuid.New().String(),
					Destination: "test://foobar0",
					Payload:     []byte("foobar payload"),
				},
				{
					ID:          ksuidTime("2019-01-01T05:05:06Z"),
					Domain:      "testing1",
					Cursor:      ksuid.New().String(),
					Destination: "test://foobar1",
					Payload:     []byte("another payload"),
				},
				{
					ID:          ksuidTime("2019-01-01T05:05:07Z"),
					Domain:      "testing2",
					Cursor:      ksuid.New().String(),
					Destination: "test://foobar2",
					Payload:     []byte("another payload hello"),
				},
			},
			test: func(ctx context.Context, t *testing.T, db destiny.EventDB, events ...destiny.Event) {
				// Read any events starting a minute before the time of the first event.
				iter := db.ReadEvents(ctx, events[0].ID.Time().Add(-1*time.Minute), time.Second)

				committed := 0
				var event destiny.Event
				for iter.Next(&event) {
					if err := event.Commit(); err != nil {
						t.Fatal(err)
					}

					assert.Equal(t, event.ID.String(), events[committed].ID.String())
					assert.Equal(t, event.Domain, events[committed].Domain)
					assert.Equal(t, string(event.Payload), string(events[committed].Payload))
					assert.Equal(t, event.Destination, events[committed].Destination)

					committed++
				}

				if err := iter.Close(); err != io.EOF {
					t.Fatalf("expected no rows error: %s", err)
				}
			},
		},

		{
			scenario: "event after offset isn't read",
			events: []destiny.Event{
				{
					ID:          ksuidTime("2019-01-01T05:05:05Z"),
					Domain:      "whoops",
					Cursor:      ksuid.New().String(),
					Destination: "test://whoops",
					Payload:     []byte("ha payload"),
				},
			},
			test: func(ctx context.Context, t *testing.T, db destiny.EventDB, events ...destiny.Event) {
				// Read any events starting a minute before the time of the first event.
				iter := db.ReadEvents(ctx, events[0].ID.Time().Add(-1*time.Second), time.Second)

				found := 0
				var event destiny.Event
				for iter.Next(&event) {
					found++
				}

				assert.Equal(t, 0, found)

				if err := iter.Close(); err != io.EOF {
					t.Fatalf("expected no rows error: %s", err)
				}
			},
		},

		{
			scenario: "read event before offset not after",
			events: []destiny.Event{
				{
					ID:          ksuidTime("2019-01-01T05:05:05Z"),
					Domain:      "whoops",
					Cursor:      ksuid.New().String(),
					Destination: "test://whoops",
					Payload:     []byte("ha payload"),
				},
				{
					ID:          ksuidTime("2019-01-01T05:05:08Z"),
					Domain:      "whoops",
					Cursor:      ksuid.New().String(),
					Destination: "test://whoops",
					Payload:     []byte("ha payload hello"),
				},
			},
			test: func(ctx context.Context, t *testing.T, db destiny.EventDB, events ...destiny.Event) {
				// Read any events starting a minute before the time of the first event.
				iter := db.ReadEvents(ctx, events[0].ID.Time().Add(1*time.Second), time.Second)

				found := 0
				var event destiny.Event
				for iter.Next(&event) {
					assert.Equal(t, events[found].ID, event.ID)
					assert.Equal(t, events[found].Payload, event.Payload)
					found++
				}

				assert.Equal(t, 1, found)

				if err := iter.Close(); err != io.EOF {
					t.Fatalf("expected no rows error: %s", err)
				}
			},
		},

		{
			scenario: "successfully commit an event after reading it",
			events: []destiny.Event{
				{
					ID:          ksuidTime("2019-01-01T05:05:05Z"),
					Domain:      "whoops",
					Cursor:      ksuid.New().String(),
					Destination: "test://whoops",
					Payload:     []byte("ha payload 555"),
				},
			},
			test: func(ctx context.Context, t *testing.T, db destiny.EventDB, events ...destiny.Event) {
				// Read any events starting a minute before the time of the first event.
				iter := db.ReadEvents(ctx, events[0].ID.Time().Add(1*time.Second), time.Second)

				var event destiny.Event
				if iter.Next(&event) == false {
					t.Fatal("expected to read an event")
				}

				if err := event.Commit(); err != nil {
					t.Fatal(err)
				}

				if err := iter.Close(); err != nil {
					t.Fatalf("unexpected error: %s", err)
				}
			},
		},

		{
			scenario: "retrying an event should re-create it with specified time",
			events: []destiny.Event{
				{
					ID:          ksuidTime("2019-01-01T05:05:05Z"),
					Domain:      "whoops",
					Cursor:      ksuid.New().String(),
					Destination: "test://whoops",
					Payload:     []byte("ha payload 555"),
				},
			},
			test: func(ctx context.Context, t *testing.T, db destiny.EventDB, events ...destiny.Event) {
				// Read any events starting a minute before the time of the first event.
				iter := db.ReadEvents(ctx, events[0].ID.Time().Add(1*time.Second), time.Second)

				var event destiny.Event
				if iter.Next(&event) == false {
					t.Fatal("expected to read an event")
				}

				if err := event.Retry(events[0].ID.Time().Add(10 * time.Minute)); err != nil {
					t.Fatal(err)
				}

				if err := iter.Close(); err != nil {
					t.Fatalf("unexpected error: %s", err)
				}

				iter = db.ReadEvents(ctx, events[0].ID.Time().Add(11*time.Minute), time.Second)

				var retryEvent destiny.Event
				if iter.Next(&retryEvent) == false {
					t.Fatal("expected to read an event")
				}

				assert.NotEqual(t, retryEvent.ID, events[0].ID)
				assert.Equal(t, events[0].Payload, retryEvent.Payload)
				assert.Equal(t, 1, retryEvent.Attempts)

				if err := retryEvent.Commit(); err != nil {
					t.Fatal(err)
				}

				if err := iter.Close(); err != nil {
					t.Fatalf("unexpected error: %s", err)
				}
			},
		},
	}

	for _, test := range tests {
		t.Run(test.scenario, func(t *testing.T) {
			ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
			defer cancel()

			db, teardown, err := makeEventDB(ctx, test.events...)
			if err != nil {
				t.Fatal(err)
			}

			defer teardown()

			test.test(ctx, t, db, test.events...)
		})
	}
}
