package twirp

import (
	"context"
	"fmt"
	"net"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"code.justin.tv/danielnf/destiny/destinytwirp"
	destiny "code.justin.tv/danielnf/destiny/internal"
	"github.com/segmentio/events"
	"github.com/segmentio/ksuid"
	"github.com/twitchtv/twirp"
)

type Destiny struct {
	eventDB destiny.EventDB
}

func (srv *Destiny) Send(ctx context.Context, req *destinytwirp.SendRequest) (res *destinytwirp.SendResponse, err error) {
	events := make([]destiny.Event, 0, len(req.Events))

	for _, event := range req.Events {
		if event.Domain == "" {
			err = twirp.InvalidArgumentError("domain", "missing or invalid domain")
			return
		}

		if event.Encoding == "" {
			err = twirp.InvalidArgumentError("encoding", "missing or invalid encoding")
			return
		}

		if event.Cursor == "" {
			event.Cursor = ksuid.New().String()
		}

		if _, err = url.Parse(event.Destination); err != nil {
			err = twirp.InvalidArgumentError("destination", "destination should be a URL (e.g., sqs://$queue)")
			return
		}

		var id ksuid.KSUID
		id, err = ksuid.NewRandomWithTime(time.Unix(event.SendAt.Seconds, int64(event.SendAt.Nanos)).UTC())
		if err != nil {
			err = twirp.InternalErrorWith(err)
			return
		}

		events = append(events, destiny.Event{
			ID:          id,
			Domain:      event.Domain,
			Destination: event.Destination,
			Payload:     event.Payload,
			Cursor:      event.Cursor,
			Encoding:    event.Encoding,
			MaxRetries:  int(event.MaxRetries),
		})
	}

	if err = srv.eventDB.WriteEvents(ctx, events...); err != nil {
		err = twirp.InternalErrorWith(err)
		return
	}

	res = &destinytwirp.SendResponse{}
	return
}

// NewHandler creates a single HTTP handler that can be used for the Destiny.
func NewHandler(destiny destinytwirp.Destiny) http.Handler {
	chronoHandler := destinytwirp.NewDestinyServer(destiny, nil)
	mux := http.NewServeMux()

	mux.HandleFunc("/", func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
		w.Write([]byte("Destiny!"))
	})

	mux.Handle(destinytwirp.DestinyPathPrefix, chronoHandler)

	events.Log("(public)  destiny twirp handler: %{handler}s", destinytwirp.DestinyPathPrefix)
	return mux
}

// NewDestiny returns an implementation of the destinytwirp.Destiny` service interface.
func NewDestiny(eventDB destiny.EventDB) *Destiny {
	if eventDB == nil {
		panic("a non-nil EventDB is required")
	}

	return &Destiny{
		eventDB: eventDB,
	}
}

func MakeChronobreak(eventDB destiny.EventDB) (destinytwirp.Destiny, func(), error) {
	return NewDestiny(eventDB), func() {}, nil
}

func MakeDestinyWithHTTP(eventDB destiny.EventDB) (destinytwirp.Destiny, func(), error) {
	destiny := NewDestiny(eventDB)

	handler := destinytwirp.NewDestinyServer(destiny, nil)
	mux := http.NewServeMux()
	mux.Handle(destinytwirp.DestinyPathPrefix, handler)

	srv, port, err := newServer(mux)
	if err != nil {
		return nil, func() {}, err
	}

	teardown := func() {
		if err := srv.Shutdown(context.Background()); err != nil {
			fmt.Printf("error closing http server: %s\n", err)
		}
	}

	client := destinytwirp.NewDestinyProtobufClient("http://localhost:"+port, &http.Client{})
	return client, teardown, nil
}

func newServer(handler http.Handler) (*http.Server, string, error) {
	listener, err := net.Listen("tcp", ":0")
	if err != nil {
		return nil, "", err
	}

	port := strconv.Itoa(listener.Addr().(*net.TCPAddr).Port)

	srv := &http.Server{
		Handler: handler,
	}

	go func() {
		if err := srv.Serve(listener); err != nil && err != http.ErrServerClosed {
			panic(err)
		}
	}()

	return srv, port, nil
}
