#!/bin/bash
#global declarations
whipdir="up"
phase=3
popver=5.0
originver=6.1
dcsize="standard"
debug=false
declare -A pduamac pdua2mac pdua3mac pdubmac pdub2mac pdub3mac ts1mac ts2mac ts3mac ts4mac pduaip pdubip pdua2ip pdub2ip pdua3ip pdub3ip ts1ip ts2ip ts3ip ts4ip


while getopts ":R:kS:d:P:s:V:ofDa" opt; do
  case ${opt} in
    R ) rack="r${OPTARG/"r"/}"
        if [ ${#rack} -eq 3 ]; then
          dctype="pop"
          if [ ${rack: -2} -eq 02 ]; then
            racktype="net1"
          elif [ ${rack: -2} -eq 03 ]; then
            racktype="net2"
          elif [ ${rack: -2} -ge 04 -a ${rack: -2} -le 10 ]; then
            racktype="server"
          else
            echo "invalid rack number: $OPTARG, please try again" 1>&2
            exit 1
          fi
        elif [ ${#rack} -eq 4 ]; then
          dctype="origin"
          if [ ${rack:1:1} -eq 1 ]; then
            case ${rack: -2} in 
              02 ) racktype="net1"
                ;;
              03 ) racktype="ds1"
                ;;
              04 ) racktype="ds2"
                ;;
              05 ) racktype="net2"
                ;;
              07 ) racktype="sec"
                ;;
              * ) echo "unknown rack $OPTARG, please try again." 1>&2
                exit 1
                ;;
            esac
          elif [ ${rack:1:1} -ge 2 ]; then
            if [ ${rack: -2} -eq 00 ]; then
              racktype="eor"
            else
              racktype="checkK"
            fi
          fi
        fi
      ;;
    S ) dc=${OPTARG^^}
      ;;
    s ) dcsize=$OPTARG
      ;;
    d ) whipdir=${OPTARG,,}
      ;;
    P ) phase=$OPTARG
      if [ $phase -ne 1 -a $phase -ne 3 ]; then
        echo "Only 1 or 3 accepted here. 1 being single-phase power, 3 being 3-phase power." 1>&2
        exit 1
      fi
      ;;
    k ) iskon=true
      ;;
    V ) version=$OPTARG
      ;;
    o ) onerack=true
      ;;
    a ) allrack=true
      ;;
    D ) debug=true
      ;;
    \? ) echo "Invalid option: $OPTARG" 1>&2
      ;;
    : ) echo "$OPTARG Requires an argument" 1>&2
  esac
done
#Print welcome message
printf "Welcome to $0! You may use Ctrl+C at any point from here to cancel and stop the software, changes should not be saved till you see 'You are past the point of no return'\n\n"


#declare some common stuffs and set some settings
#if version is not provided, use latest known
if [ -z $version ]; then
  if [ "${dctype}" = "origin" ]; then
    version=$originver
  elif [ "${dctype}" = "pop" ]; then
    version=$popver
  else
    echo "Incorrect DC Type. You should not see this message. Contact Aloisuz." 1>&2
    exit 1
  fi
fi

#check if user is confirm its a Kontron Chassis
if [ "${racktype}" = "checkK" ]; then
  if [ $iskon ]; then
    racktype="kon"
  else
    echo "Only kontrons are supported in Origins (as only kontron racks has TSes and PDUs that require configurations. If you intend to provison a Kontron, please specify -k as one of the arguments." 1>&2
    exit 1
  fi
fi

#set rack clases and some rack definations
case $racktype in
  kon ) rackclass="kon"
      numts=1
      phase=3
    ;;
  eor ) rackclass="eor"
      numts=4
      phase=1
    ;;
  net1 ) rackclass="net"
    ;;
  net2 ) rackclass="net"
    ;;
  ds1 ) rackclass="net"
    ;;
  ds2 ) rackclass="net"
    ;;
  sec ) rackclass="sec"
      numts=0
    ;;
  server ) rackclass="svr"
      numts=0
    ;;
esac

#Pull subnet from IB
#check if LDAP Creds has been exported. If not exported then ask for it
if [ "${ldapcreds}" = "" ]; then
  read -p "LDAP Username: " ldapuser
  read -s -p "LDAP Password: " ldappass
  ldapcreds=$ldapuser:$ldappass
  printf "\n"
fi
#check DC Type, set query string for IB
if [ "${dctype}" = "pop" -a "${rackclass}" = "svr" ]; then
  commentstr="$dc - $rack Management"
elif [ "${dctype}" = "pop" -a "${rackclass}" = "net" ]; then
  commentstr="$dc - Network Racks 1-3 (r01-r03) Management"
elif [ "${dctype}" = "origin" ]; then
  if [ "${rackclass}" = "net" -o "${rackclass}" = "sec" ]; then  
    commentstr="r100-ma01-ma02.${dc,,}"
  elif [ "${rackclass}" = "kon" ]; then
    commentstr="$rack-ma01-ma02.${dc,,}"
  fi
fi
#query IB for subnet
echo "Grabbing subnet from InfoBlox..."
result=$(curl -k -u "$ldapcreds" -X GET 'https://infoblox.live-video.a2z.com/wapi/v2.11.1/network' -d comment="$commentstr" -d _return_fields="network,options" 2>/dev/null)
if [[  $result == *"401 Authorization Required"* ]]; then
  echo "ERROR: Incorrect LDAP Creds. Please check LDAP Creds and try again." 1>&2
  exit 1
elif [[  $result == "[]" ]]; then
  echo "ERROR: Network not found, please ensure network is created in IB. If records don't exist in IB, we won't be able to grab the network details." 1>&2
  exit 1
fi
subnet=$(echo $result | jq -r '.[]|.network')
gateway=$(echo $result | jq -r '.[]|.options|.[]|select(.name=="routers")|.value')
echo "Got subnet: $subnet"
echo "Gateway: $gateway"

#assign rack classes & quantity of TSes/PDUs & IP Addresses
#set default IP addresses for TS gateway
#ts1ip=$(echo $subnet | cut -d '.' -f 1-3)"."50
#ts2ip=$(echo $subnet | cut -d '.' -f 1-3)"."100
#ts3ip=$(echo $subnet | cut -d '.' -f 1-3)"."150
#ts4ip=$(echo $subnet | cut -d '.' -f 1-3)"."200
tsgw=$gateway

#Check rack class, define IP Addresses accordingly.
if [ "${rackclass}" = kon -o "${rackclass}" = "eor" ]; then
  ts1ip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."50
  ts2ip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."100
  ts3ip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."150
  ts4ip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."200
  pduaip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."10
  pdubip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."11
  if [ "${rackclass}" = kon ]; then
    ts1ip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."200
  fi
elif [ "${rackclass}" = "net" -a "${dctype}" = "pop" ]; then
  ts1ip[02]=$(echo $subnet | cut -d '.' -f 1-3)"."150
  ts1ip[03]=$(echo $subnet | cut -d '.' -f 1-3)"."200
  pduaip[02]=$(echo $subnet | cut -d '.' -f 1-3)"."10
  pdua2ip[02]=$(echo $subnet | cut -d '.' -f 1-3)"."11
  pdubip[02]=$(echo $subnet | cut -d '.' -f 1-3)"."12
  pdub2ip[02]=$(echo $subnet | cut -d '.' -f 1-3)"."13
  pduaip[03]=$(echo $subnet | cut -d '.' -f 1-3)"."14
  pdua2ip[03]=$(echo $subnet | cut -d '.' -f 1-3)"."15
  pdubip[03]=$(echo $subnet | cut -d '.' -f 1-3)"."16
  pdub2ip[03]=$(echo $subnet | cut -d '.' -f 1-3)"."17
elif [ "${rackclass}" = "svr" -a "${dctype}" = "pop" ]; then
  pduaip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."132
  pdua2ip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."133
  pdua3ip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."134
  pdubip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."135
  pdub2ip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."136
  pdub3ip[${rack/"r"/}]=$(echo $subnet | cut -d '.' -f 1-3)"."137
elif [ "${rackclass}" = "net" -a "${dctype}" = "origin" ]; then
  ts1ip[102]=$(echo $subnet | cut -d '.' -f 1-3)"."150
  ts1ip[105]=$(echo $subnet | cut -d '.' -f 1-3)"."200
  pduaip[102]=$(echo $subnet | cut -d '.' -f 1-3)"."10
  pdua2ip[102]=$(echo $subnet | cut -d '.' -f 1-3)"."11
  pdubip[102]=$(echo $subnet | cut -d '.' -f 1-3)"."12
  pdub2ip[102]=$(echo $subnet | cut -d '.' -f 1-3)"."13
  pduaip[103]=$(echo $subnet | cut -d '.' -f 1-3)"."14
  pdua2ip[103]=$(echo $subnet | cut -d '.' -f 1-3)"."15
  pdubip[103]=$(echo $subnet | cut -d '.' -f 1-3)"."16
  pdub2ip[103]=$(echo $subnet | cut -d '.' -f 1-3)"."17
  pduaip[104]=$(echo $subnet | cut -d '.' -f 1-3)"."18
  pdua2ip[104]=$(echo $subnet | cut -d '.' -f 1-3)"."19
  pdubip[104]=$(echo $subnet | cut -d '.' -f 1-3)"."20
  pdub2ip[104]=$(echo $subnet | cut -d '.' -f 1-3)"."21
  pduaip[105]=$(echo $subnet | cut -d '.' -f 1-3)"."22
  pdua2ip[105]=$(echo $subnet | cut -d '.' -f 1-3)"."23
  pdubip[105]=$(echo $subnet | cut -d '.' -f 1-3)"."24
  pdub2ip[105]=$(echo $subnet | cut -d '.' -f 1-3)"."25
fi

#check rack class, ask if user wants to provision the full network rack or a single rack unless -o is specified
if [ "${rackclass}" = "net" ]; then
  if [ ! "$onerack" -a ! "$allrack" ]; then
    printf "Network row detected. Would you like to setup all the network rack PDUs and TSes or just $rack?\n"
    select choice in "All" "$rack" "Cancel"; do
      case $choice in
        All ) echo "You have chosen to setup all the racks. Brace yourself for data entry of mac addresses!!"
          if [ "${dctype}" = "pop" ]; then
            rackstogothrough=(02 03)
          elif [ "${dctype}" = "origin" ]; then
            rackstogothrough=(102 103 104 105)
          fi
          break
          ;;
        $rack ) echo "You have chosen to setup $rack."
          rackstogothrough=(${rack/"r"/})
          break
          ;;
        Cancel ) echo "You have chosen to cancel. Exiting now."
          exit 0
          ;;
      esac
    done
  elif [ "$onerack" = true ] ; then
    rackstogothrough=(${rack/"r"/})
  elif [ "$allrack" = true ] ; then
    if [ "${dctype}" = "pop" ]; then
      rackstogothrough=(02 03)
    elif [ "${dctype}" = "origin" ]; then
      rackstogothrough=(102 103 104 105)
    fi
  fi
else
  rackstogothrough=(${rack/"r"/})
fi

#Force numpdu to 2 if phase=3 or if Rack Class is eor. Else, check DC Type, Rack Class and Phase to determine numpdu
if [ $phase -eq 3 -o "${rackclass}" = "eor" ]; then
  numpdu=2
elif [ $phase -eq 1 ]; then
  if [ "${rackclass}" = "net" ]; then
    numpdu=4
  elif [ "${rackclass}" = "svr" ]; then
    if [ $phase -eq 1 ]; then
      numpdu=6
    fi
  fi
fi

if [ "${whipdir}" = "up" ]; then
  whipdir="WhipUp"
else
  whipdir="WhipDown"
fi

#read MAC Addresses
for r in "${rackstogothrough[@]}"
do
  if [ "${rackclass}" = net ]; then
    if [ $r = 102 -o $r = 105 ]; then
      numts=1
    elif [ $r = 02 -o $r = 03 ]; then
      numts=1
    else
      numts=0
    fi
  fi

  read -p "Enter pdu${r}a MAC Address and press [ENTER]: " pduamac[${r}]
  read -p "Enter pdu${r}b MAC Address and press [ENTER]: " pdubmac[${r}]
  if [ $numpdu -ge 4 -o $numpdu -eq 6 ]; then
    read -p "Enter pdu${r}a2 MAC Address and press [ENTER]: " pdua2mac[${r}]
    read -p "Enter pdu${r}b2 MAC Address and press [ENTER]: " pdub2mac[${r}]
  fi
  if [ $numpdu -eq 6 ]; then
    read -p "Enter pdu${r}a3 MAC Address and press [ENTER]: " pdua3mac[${r}]
    read -p "Enter pdu${r}b3 MAC Address and press [ENTER]: " pdubb3mac[${r}]
  fi
  
  if [ $numts -gt 0 ]; then
    read -p "Enter r${r}-ts01 MAC Address and press [ENTER]: " ts1mac[${r}]
  fi
  if [ $numts -ge 2 -a $numts -le 4 ]; then
    read -p "Enter r${r}-ts02 MAC Address and press [ENTER]: " ts2mac[${r}]
  fi
  if [ $numts -ge 3 -a $numts -le 4 ]; then
    read -p "Enter r${r}-ts03 MAC Address and press [ENTER]: " ts3mac[${r}]
  fi
  if [ $numts -eq 4 ]; then
    read -p "Enter r${r}-ts04 MAC Address and press [ENTER]: " ts4mac[${r}]
  fi
done

#This part handles the actual generation of configuration files & build the IB Query.
#pull config from git and make new branch
printf "\n\nStarting to generate the actual configurations. After this point changes will be made to your home dir. All files are worked on in ~/cgwd and deleted after.\n\n"
mkdir -p ~/cgwd
cd ~/cgwd
git clone git@git.xarth.tv:dcops/dco_device-deployment-files.git ~/cgwd
git checkout -b ConfigGen-$dc.$rack

#generate configs for TSes and PDUs
body="["
for r in "${rackstogothrough[@]}"
do
#recheck racktype per run
  if [ "${rackclass}" = "net" ]; then
    case $r in 
      102 ) racktype="net1"
          ;;
      103 ) racktype="ds1"
          ;;
      104 ) racktype="ds2"
          ;;
      105 ) racktype="net2"
          ;;
      02 ) racktype="net1"
          ;;
      03 ) racktype="net2"
          ;;
    esac
 fi
#generate TS configs & build body of IB Request
  if [ $numts -gt 0 ]; then
     sed -e "s/<devIP>/${ts1ip[$r]}/g" -e "s/<site code>/${dc,,}/g" -e "s/<rackNum>/r${r}/g" -e "s/<gateway>/${tsgw}/g" -e "s/<devid>/ts01/g" -e "s/<dctype>/${dctype}/g" config_info_sheet/template_${dctype}_${dcsize}_${version}_${racktype}_ts01 > config_info_sheet/twitch_${ts1ip[$r]}
     if [ "${body}" != "[" ]; then
      body=$body","
     fi
     body=$body"{\"method\":\"POST\",\"object\":\"record:host\",\"data\":{\"name\":\"r${r}-ts01.${dc,,}.justin.tv\",\"ipv4addrs\":[{\"ipv4addr\":\"${ts1ip[$r]}\",\"mac\":\"${ts1mac[$r]}\",\"configure_for_dhcp\":true}]},\"assign_state\":{\"r${r}-ts01_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true}"
    for i in {1..48}
    do
      portip=$(echo ${ts1ip[$r]} | cut -d '.' -f 1-3)"."$(($(echo ${ts1ip[$r]}| cut -d '.' -f 4) + $i))
      body=$body",{\"method\":\"POST\",\"object\":\"record:a\",\"data\":{\"name\":\"s${i}.r${r}-ts01.${dc,,}.justin.tv\",\"ipv4addr\":\"$portip\"},\"assign_state\":{\"s${i}.r${r}-ts01_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true}"
    done
  fi
  if [ $numts -ge 2 -a $numts -le 4 ]; then
    sed -e "s/<devIP>/${ts1ip[$r]}/g" -e "s/<site code>/${dc,,}/g" -e "s/<rackNum>/r${r}/g" -e "s/<gateway>/${tsgw}/g" -e "s/<devid>/ts02/g" -e "s/<dctype>/${dctype}/g" config_info_sheet/template_${dctype}_${dcsize}_${version}_${racktype}_ts02 > config_info_sheet/twitch_${ts2ip[$r]}
    body=$body",{\"method\":\"POST\",\"object\":\"record:host\",\"data\":{\"name\":\"r${r}-ts02.${dc,,}.justin.tv\","ipv4addrs":[{\"ipv4addr\":\"${ts2ip[$r]}\",\"mac\":\"${ts2mac[$r]}\",\"configure_for_dhcp\":true}]},\"assign_state\":{\"r${r}-ts02_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true}"
    for i in {1..48}
    do
      portip=$(echo ${ts2ip[$r]} | cut -d '.' -f 1-3)"."$(($(echo ${ts2ip[$r]}| cut -d '.' -f 4) + $i))
      body=$body",{\"method\":\"POST\",\"object\":\"record:a\",\"data\":{\"name\":\"s${i}.r${r}-ts02.${dc,,}.justin.tv\",\"ipv4addr\":\"$portip\"},\"assign_state\":{\"s${i}.r${r}-ts02_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true}"
    done
  fi
  if [ $numts -ge 3 -a $numts -le 4 ]; then
    sed -e "s/<devIP>/${ts1ip[$r]}/g" -e "s/<site code>/${dc,,}/g" -e "s/<rackNum>/r${r}/g" -e "s/<gateway>/${tsgw}/g" -e "s/<devid>/ts03/g" -e "s/<dctype>/${dctype}/g" config_info_sheet/template_${dctype}_${dcsize}_${version}_${racktype}_ts03 > config_info_sheet/twitch_${ts3ip[$r]}
    body=$body",{\"method\":\"POST\",\"object\":\"record:host\",\"data\":{\"name\":\"r${r}-ts03.${dc,,}.justin.tv\",\"ipv4addrs\":[{\"ipv4addr\":\"${ts3ip[$r]}\",\"mac\":\"${ts3mac[$r]}\",\"configure_for_dhcp\":true}]},\"assign_state\":{\"r${r}-ts03_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true}"
    for i in {1..48}
    do
      portip=$(echo ${ts3ip[$r]} | cut -d '.' -f 1-3)"."$(($(echo ${ts3ip[$r]}| cut -d '.' -f 4) + $i))
      body=$body",{\"method\":\"POST\",\"object\":\"record:a\",\"data\":{\"name\":\"s${i}.r${r}-ts03.${dc,,}.justin.tv\",\"ipv4addr\":\"$portip\"},\"assign_state\":{\"s${i}.r${r}-ts03_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true}"
    done
  fi
  if [ $numts -eq 4 ]; then
    sed -e "s/<devIP>/${ts1ip[$r]}/g" -e "s/<site code>/${dc,,}/g" -e "s/r<rackNum>/${r}/g" -e "s/<gateway>/${tsgw}/g" -e "s/<devid>/ts04/g" -e "s/<dctype>/${dctype}/g" config_info_sheet/template_${dctype}_${dcsize}_${version}_${racktype}_ts04 > config_info_sheet/twitch_${ts4ip[$r]}
    body=$body",{\"method\":\"POST\",\"object\":\"record:host\",\"data\":{\"name\":\"r${r}-ts04.${dc,,}.justin.tv\",\"ipv4addrs\":[{\"ipv4addr\":\"${ts4ip[$r]}\",\"mac\":\"${ts4mac[$r]}\",\"configure_for_dhcp\":true}]},\"assign_state\":{\"r${r}-ts04_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true}"
    for i in {1..48}
    do
      portip=$(echo ${ts4ip[$r]} | cut -d '.' -f 1-3)"."$(($(echo ${ts4ip[$r]}| cut -d '.' -f 4) + $i))
      body=$body",{\"method\":\"POST\",\"object\":\"record:a\",\"data\":{\"name\":\"s${i}.r${r}-ts04.${dc,,}.justin.tv\",\"ipv4addr\":\"$portip\"},\"assign_state\":{\"s${i}.r${r}-ts04_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true}"
    done
  fi

#generate PDU configs & Body of IB Request
  cat pdu_configs/templates/pdu_common | tee pdu_configs/pdu_${pduamac[$r]//":"/}.ini pdu_configs/pdu_${pdubmac[$r]//":"/}.ini > /dev/null
  sed -e "s/<site code>/${dc}/g" -e "s/<rackNum>/${r}/g" -e "s/<sn>//g" pdu_configs/templates/pdu_a_template >> pdu_configs/pdu_${pduamac[$r]//":"/}.ini
  sed -e "s/<site code>/${dc}/g" -e "s/<rackNum>/${r}/g" -e "s/<sn>//g" pdu_configs/templates/pdu_b_template >> pdu_configs/pdu_${pdubmac[$r]//":"/}.ini
  cat pdu_configs/templates/$whipdir/template_${dctype}_${dcsize}_${version}_${phase}p_${racktype}_a >> pdu_configs/pdu_${pduamac[$r]//":"/}.ini
  cat pdu_configs/templates/$whipdir/template_${dctype}_${dcsize}_${version}_${phase}p_${racktype}_b >> pdu_configs/pdu_${pdubmac[$r]//":"/}.ini
  if [ "${body}" != "[" ]; then
      body=$body","
  fi
  body=$body"{\"method\":\"POST\",\"object\":\"record:host\",\"data\":{\"name\":\"pdu${r}a.${dc,,}.justin.tv\",\"ipv4addrs\":[{\"ipv4addr\":\"${pduaip[$r]}\",\"mac\":\"${pduamac[$r]}\",\"configure_for_dhcp\":true}]},\"assign_state\":{\"pdu${r}a_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true},{\"method\":\"POST\",\"object\":\"record:host\",\"data\":{\"name\":\"pdu${r}b.${dc,,}.justin.tv\",\"ipv4addrs\":[{\"ipv4addr\":\"${pdubip[$r]}\",\"mac\":\"${pdubmac[$r]}\",\"configure_for_dhcp\":true}]},\"assign_state\":{\"pdu${r}b_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true}"
  if [ $numpdu -ge 4 -o $numpdu -eq 6 ]; then
    cat pdu_configs/templates/pdu_common | tee pdu_configs/pdu_${pdua2mac[$r]//":"/}.ini pdu_configs/pdu_${pdub2mac[$r]//":"/}.ini > /dev/null
    sed -e "s/<site code>/${dc}/g" -e "s/<rackNum>/${r}/g" -e "s/<sn>/2/g" pdu_configs/templates/pdu_a_template >> pdu_configs/pdu_${pdua2mac[$r]//":"/}.ini
    sed -e "s/<site code>/${dc}/g" -e "s/<rackNum>/${r}/g" -e "s/<sn>/2/g" pdu_configs/templates/pdu_b_template >> pdu_configs/pdu_${pdub2mac[$r]//":"/}.ini
    cat pdu_configs/templates/$whipdir/template_${dctype}_${dcsize}_${version}_${phase}p_${racktype}_a2 >> pdu_configs/pdu_${pdua2mac[$r]//":"/}.ini
    cat pdu_configs/templates/$whipdir/template_${dctype}_${dcsize}_${version}_${phase}p_${racktype}_b2 >> pdu_configs/pdu_${pdub2mac[$r]//":"/}.ini
    body=$body",{\"method\":\"POST\",\"object\":\"record:host\",\"data\":{\"name\":\"pdu${r}a2.${dc,,}.justin.tv\",\"ipv4addrs\":[{\"ipv4addr\":\"${pdua2ip[$r]}\",\"mac\":\"${pdua2mac[$r]}\",\"configure_for_dhcp\":true}]},\"assign_state\":{\"pdu${r}a2_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true},{\"method\":\"POST\",\"object\":\"record:host\",\"data\":{\"name\":\"pdu${r}b2.${dc,,}.justin.tv\",\"ipv4addrs\":[{\"ipv4addr\":\"${pdub2ip[$r]}\",\"mac\":\"${pdub2mac[$r]}\",\"configure_for_dhcp\":true}]},\"assign_state\":{\"pdu${r}b2_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true}"
  elif [ $numpdu -eq 6 ]; then
    cat pdu_configs/templates/pdu_common | tee pdu_configs/pdu_${pdua3mac[$r]//":"/}.ini pdu_configs/pdu_${pdub3mac[$r]//":"/}.ini > /dev/null
    sed -e "s/<site code>/${dc}/g" -e "s/<rackNum>/${r}/g" -e "s/<sn>/3/g" pdu_configs/templates/pdu_a_template >> pdu_configs/pdu_${pdua3mac[$r]//":"/}.ini
    sed -e "s/<site code>/${dc}/g" -e "s/<rackNum>/${r}/g" -e "s/<sn>/3/g" pdu_configs/templates/pdu_b_template >> pdu_configs/pdu_${pdub3mac[$r]//":"/}.ini
    cat pdu_configs/templates/$whipdir/template_${dctype}_${dcsize}_${version}_${phase}p_${racktype}_a3 >> pdu_configs/pdu_${pdua3mac[$r]//":"/}.ini
    cat pdu_configs/templates/$whipdir/template_${dctype}_${dcsize}_${version}_${phase}p_${racktype}_b3 >> pdu_configs/pdu_${pdub3mac[$r]//":"/}.ini
    body=$body",{\"method\":\"POST\",\"object\":\"record:host\",\"data\":{\"name\":\"pdu${r}a3.${dc,,}.justin.tv\",\"ipv4addrs\":[{\"ipv4addr\":\"${pdua3ip[$r]}\",\"mac\":\"${pdua3mac[$r]}\",\"configure_for_dhcp\":true}]},\"assign_state\":{\"pdu${r}a3_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true},{\"method\":\"POST\",\"object\":\"record:host\",\"data\":{\"name\":\"pdu${r}b3.${dc,,}.justin.tv\",\"ipv4addrs\":[{\"ipv4addr\":\"${pdub3ip[$r]}\",\"mac\":\"${pdub3mac[$r]}\",\"configure_for_dhcp\":true}]},\"assign_state\":{\"pdu${r}b3_ref\":\"_ref\"},\"enable_substitution\":true,\"discard\":true}"
  fi

done
body=$body"]"


#commit changes to git and remove directories
git add -A
git commit -m "Added $dc $rack configs"
if ! $debug ; then
  git checkout master
  git merge ConfigGen-$dc.$rack
  printf "\n\nYou are past the point of no return. Ctrl+C won't save you now.\n\n"
  git branch -d ConfigGen-$dc.$rack
  git push
fi
rm -rf ~/cgwd

#send IB Request
echo "Creating hosts in InfoBlox..."
if ! $debug ; then
  result=$(curl --silent -H 'content-type:application/json' -k -u "$ldapcreds" -X POST 'https://infoblox.live-video.a2z.com/wapi/v2.11.1/request' -d "$body" 2>/dev/null)
  echo $result
  curl --silent -k -u "$ldapcreds" -H 'content-type: application/json' -X POST https://infoblox.live-video.a2z.com/wapi/v2.11.1/grid/b25lLmNsdXN0ZXIkMA:Twitch?_function=restartservices -d '{"restart_option":"RESTART_IF_NEEDED","service_option":"ALL","member_order":"SEQUENTIALLY","sequential_delay":10}'
else
   echo $body
fi
