import paramiko, time, getpass, argparse, re, threading, sys, subprocess, socket
# import ipaddress


###############################################################################
## Class Definitions ##########################################################
###############################################################################

class kontronChassis:
	site = None
	rack = None
	number = None
	uLevel = None
	serial = None
	fqdn = None
	hubnodes = [] # Use of list does not pre-dispose number of hubnodes, but invesitation function does
	nodes = []
	active_hubnode = None
	sdr_data = [] # This will most likely be a list of lines of 'sdr elist' output
	sel_data = []
	investigation_loading_bar = None
	investigation_hubnode_target = 14
	investigation_hubnode_progress = 0
	investigation_node_target = 54
	investigation_node_progress = 0
	investigation_fail_through = False
	dns_error = False
	


	def __init__(self,site,rack,chassisNo):
		self.site = site
		self.rack = 'r'+re.sub('[^0-9]','',str(rack))
		self.number = chassisNo
		self.fqdn = str(self.rack)+'-kontron'+str(self.number)+'.'+str(self.site)+'.justin.tv'

	def __str__(self):
		return "kontronChassis<"+str(self.fqdn)+">"

	def investigate(self,console_username,console_password,verbose=False): # Note - Assumes 2 hubnodes and 9 nodes
		self.hubnodes.append(kontronHubnode(self,1))
		self.hubnodes.append(kontronHubnode(self,2))

		if self.dns_error:
			if verbose: print('ERROR: error found when attempting to resolve procedurally generated DNS names')
		else:
			### start threaded process - number of thread linear to number of hubnodes, for wider application implement max thread list size; create thread count variable to replace len() in the thread creation loop
			check_threads=[]
			for i in range(len(self.hubnodes)):
				check_threads.append(threading.Thread(target=self.hubnodes[i].investigate, args=(console_username,console_password,verbose)))
			for i in range(len(self.hubnodes)):
				check_threads[i].start()
			for i in range(len(self.hubnodes)):
				check_threads[i].join()
			### end threaded process
			#self.investigation_hubnode_progress = self.investigation_hubnode_target

			for i in range(1,10): # 'i' is the nod ename, starting at 1, index will be i-1
				self.nodes.append(kontronNode(self,i))
			if self.active_hubnode:
				if self.active_hubnode.ipmi_ip is not None:
					for i in range(1,10): # 'i' is the nod ename, starting at 1, index will be i-1
						#self.nodes.append(kontronNode(self,i))
						# Note that the node investigate function uses targeted requests from the active hubnode, do not multi-thread this function for the same chassis
						self.nodes[i-1].investigate(verbose)
			#self.investigation_node_progress = self.investigation_node_target

	def check(self):
		warnings = dict()
		for line in self.sel_data:
			errors_list = ['OEM','Platform Alert']
			if any(x in line for x in errors_list):
				#warnings['Event Log'].append("OEM Firmware Event found in LOG: "+str(line[1])+" "+str(line[2])+" "+str(line[3])+" - "+str(line[5])+" - "+str(line[6])+" "+str(line[7]))
				temp_desc = 'Found: '+str(line.split('|')[5])

				if 'SEL' not in warnings:
					warnings['SEL']=[]
				if temp_desc not in warnings['SEL']:
					warnings['SEL'].append(temp_desc)
		for hubnode in self.hubnodes:
			if hubnode.serial == None:
				if 'Hubnode'+str(hubnode.number) not in warnings:
					warnings['Hubnode'+str(hubnode.number)]=[]
				warnings['Hubnode'+str(hubnode.number)].append("Serial Not Found")

			if hubnode.ipmi_ip == None:
				if 'Hubnode'+str(hubnode.number) not in warnings:
					warnings['Hubnode'+str(hubnode.number)]=[]
				warnings['Hubnode'+str(hubnode.number)].append("IPMI Address Not Found")
			elif hubnode.ipmi_ip.strip() == '0.0.0.0':
				if 'Hubnode'+str(hubnode.number) not in warnings:
					warnings['Hubnode'+str(hubnode.number)]=[]
				warnings['Hubnode'+str(hubnode.number)].append("IPMI Address found as "+str(hubnode.ipmi_ip))
			# warnings['Hubnode'+str(hubnode.number)].append("M State is "+str(hubnode.m_state))
		if len(self.nodes) == 0:
			warnings['All Nodes']=['No Nodes Found - Possible issue with expected active hubnode']
		for node in self.nodes:
			if node.serial == None:
				if 'Node'+str(node.number) not in warnings:
					warnings['Node'+str(node.number)]=[]
				warnings['Node'+str(node.number)].append("Serial Not Found")
			if node.ipmi_ip == None:
				if 'Node'+str(node.number) not in warnings:
					warnings['Node'+str(node.number)]=[]
				warnings['Node'+str(node.number)].append("IPMI Address Not Found")
			elif node.ipmi_ip.strip() == '0.0.0.0':
				if 'Node'+str(node.number) not in warnings:
					warnings['Node'+str(node.number)]=[]
				warnings['Node'+str(node.number)].append("IPMI Address found as "+str(node.ipmi_ip))
		return warnings

	def printTable(self):
		result_string=''
		column_width=15
		result_string='|'.join(str(x).center(column_width) for x in ['Name','Present','Serial','IPMI IP','M State'])
		for i, d in enumerate(self.hubnodes):
			temp_presnce = ''
			if d.active:
				temp_presnce = 'Active'
			else:
				temp_presnce = 'Standby'
			printLine = '|'.join(str(x).center(column_width) for x in ['hubnode'+str(d.number),temp_presnce,d.serial,d.ipmi_ip,d.m_state])
			if i == 0:
				result_string=result_string+'\n'+('-' * len(printLine))
			result_string=result_string+'\n'+printLine
		for i, d in enumerate(self.nodes):
			printLine = '|'.join(str(x).center(column_width) for x in ['node'+str(d.number),d.present,d.serial,d.ipmi_ip,d.m_state])
			if i == 0:
				result_string=result_string+'\n'+('-' * len(printLine))
			result_string=result_string+'\n'+printLine
		return result_string




class kontronHubnode:
	parent = None
	# target = None # Probably not needed, target 0x20 is relative to hubnode servicing the request
	number = None
	active = None
	serial = None
	m_state = None
	ipmi_ip = None
	ipmi_mac = None
	serial_fqdn = None
	
	# Define initialization of object
	def __init__(self,chassis,number):
		self.number = number
		self.parent = chassis
		if self.number == 1:
			self.serial_fqdn = 'con.'+str(self.parent.fqdn)
		elif self.number == 2:
			self.serial_fqdn = 'con2.'+str(self.parent.fqdn)
		if not hostname_resolves(self.serial_fqdn): self.parent.dns_error = True

	
	# Define string representation of object
	def __str__(self):
		return "kontronHubnode<hubnode"+str(self.number)+"-"+str(self.parent)+">"

	# Function to investigate a hubnode
	def investigate(self,console_username,console_password,verbose=False):
		#######################################################################
		## Console Insepction of base data via TS #############################
		#######################################################################
		self.parent.investigation_hubnode_progress += 1
		console_data = hubnode_console_inspect_base(console_username,console_password,self.serial_fqdn,22,verbose)
		#print(console_data
		if 'ERROR' in console_data.keys():
			self.parent.investigation_fail_through=True
		else:
			self.parent.investigation_hubnode_progress += 3
			if verbose:
				print('Console serial data:')
				print(console_data)
			# Check for active status, if active, set chassis serial number
			if len(console_data['Chassis Serial']) >= 1:
				self.parent.serial=console_data['Chassis Serial'][0]
				self.active=True
			else:
				self.active=False
			# Set object data
			self.serial=console_data['Board Serial'][0]
			self.ipmi_ip=console_data['IP Address'][0]
			self.ipmi_mac=console_data['MAC Address'][0]

		#######################################################################
		## IPMI Inspection of hubnode #########################################
		#######################################################################
		if self.ipmi_ip is not None:
			# grab 'sdr elist' output
			sdr_temp_output = run_IPMI_Command_IOL(self.ipmi_ip,"0x20","sdr elist", verbose).splitlines()
			self.parent.investigation_hubnode_progress += 2
			# set m_state based on 'FRU0 Hot Swap' output which is specific to queried hubnode
			self.m_state=[x for x in sdr_temp_output if re.search('FRU0 Hot Swap',x)][0].split('|')[4].strip().split(' ')[2]
			# only update active sdr info for chassis if this is the active hubnode
			if self.active:
				self.parent.sdr_data = sdr_temp_output
				self.parent.active_hubnode = self
				self.parent.sel_data = run_IPMI_Command_IOL(self.ipmi_ip,"0x20","sel elist last 200",verbose).splitlines()
				self.parent.investigation_hubnode_progress += 2



class kontronNode:
	parent = None
	target = None
	number = None #Note, numbers should range 1-9 for standard 2U kontron chassis
	serial = None
	m_state = None
	ipmi_ip = None
	ipmi_mac = None
	product = None
	part_number = None
	mfg_date = None
	present = None
	sdr_data = []
	node_target_list = ["0x82","0x84","0x86","0x88","0x8A","0x8C","0x8E","0x90","0x92"]

	# Define initialization of object
	def __init__(self,chassis,number):
		self.number = number
		self.parent = chassis
		self.target = self.node_target_list[number-1]

	# Define string representation of object
	def __str__(self):
		return "kontronNode<node"+str(self.number)+"-"+str(self.parent)+">"

	# Function to investigate a noce, this assumes IOL credentials as default for hubnode connectivity
	def investigate(self,verbose=False):

		# Use parent SDR date to get Prenence and M State
		self.m_state=[x for x in self.parent.sdr_data if re.search(str('Node'+str(self.number)+' Hot Swap'),x)][0].split('|')[4].strip().split(' ')[2]
		if verbose: print('Found m state for Node '+str(self.number)+': '+str(self.m_state))
		if [x for x in self.parent.sdr_data if re.search(str('Node'+str(self.number)+':Present'),x)][0].split('|')[4].strip() == "Present": self.present=True

		# Gather SDR information from node (this includes overall sled M State like the hubnode's output but also has the paylode M States)
		if verbose: print('Running IPMI command against '+str(self.parent.active_hubnode.ipmi_ip)+' at target '+str(self.target)+': sdr elist')
		self.sdr_data=run_IPMI_Command_IOL(self.parent.active_hubnode.ipmi_ip,self.target,"sdr elist",verbose).splitlines()
		self.parent.investigation_node_progress += 2

		# Gather FRU information from node
		if verbose: print('Running IPMI command against '+str(self.parent.active_hubnode.ipmi_ip)+' at target '+str(self.target)+': fru')
		fru_output=run_IPMI_Command_IOL(self.parent.active_hubnode.ipmi_ip,self.target,"fru",verbose).splitlines()
		self.parent.investigation_node_progress += 2

		# Gather lan print information from all nodes via IOL
		if verbose: print('Running IPMI command against '+str(self.parent.active_hubnode.ipmi_ip)+' at target '+str(self.target)+': lan print')
		lan_output=run_IPMI_Command_IOL(self.parent.active_hubnode.ipmi_ip,self.target,"lan print",verbose).splitlines()
		self.parent.investigation_node_progress += 2
		
		# Find lines in output that match regex, produces list
		# Remove elements from list based if including 'Source'
		self.ipmi_ip=[x for x in [x for x in lan_output if re.search('IP Address',x)] if 'Source' not in x][0].split(':')[1].strip()
		self.serial=[x for x in fru_output if re.search('Serial',x)][0].split(':')[1].strip()
		self.part_number=[x for x in fru_output if re.search('Board Part Number',x)][0].split(':')[1].strip()
		self.mfg_date=[x for x in fru_output if re.search('Board Mfg Date',x)][0].split(':')[1].strip()
		self.product=[x for x in fru_output if re.search('Board Product',x)][0].split(':')[1].strip()

		# Parsing Logic
		# 1 - IPMI command run
		# 2 - IPMI command output split into list of lines - splitlines()
		# 3 - Create new list by pulling out elements of list matching regex - [x for x in <list> if re.search(<regex>,x)]
		#     note that nested search or more complex regex can be used to exclude elements
		# 4 - Take first result from newly created list - list[0]
		# 5 - Split resulting line on delimter used in IPMI output (':' or '|') into a list - result.split(':')
		# 6 - Take desired result index from the new list - list[1]
		# 7 - Clean final value of leading and trailing whitespace - value.strip()
		# 8 - Resulting string is a cleaned version of the desired value



###############################################################################
## Function Definitions #######################################################
###############################################################################



def run_IPMI_Command_IOL(host,target,command,verbose = False):
	# This is specific to the kontron version of ipmitool in the environment supporting this script
	# Update the absolute path of the tool within the .run() command
	if verbose: print("IPMI command sent to %s targeting %s with command: %s" % (host,target,command))
	process = subprocess.run("/home/users/rwilkinson/code/dco_scripts/kontron/kontron-ipmitool-1.8.13-K11/src/ipmitool -H "+str(host)+" -U admin -P admin -I lanplus -t "+str(target)+" "+str(command),shell=True, check=True, stdout=subprocess.PIPE, universal_newlines=True)
	return process.stdout

def printProgressBar (iteration, total, prefix = '', suffix = '', decimals = 1, length = 100, fill = '█', printEnd = "\r"):
    percent = ("{0:." + str(decimals) + "f}").format(100 * (iteration / float(total)))
    filledLength = int(length * iteration // total)
    bar = fill * filledLength + '-' * (length - filledLength)
    print('\r%s |%s| %s%% %s' % (prefix, bar, percent, suffix), end = printEnd)
    # Print New Line on Complete
    # if iteration == total: 
    #   print()

def loading_bar(chassis_object):
	# Loading bar for hubnode checks
	print()
	counter_elapsed = 0 # This is counter of status checks to progress thread if there is any failures that isn't caught
	printProgressBar(0, chassis_object.investigation_hubnode_target, prefix = 'Hubnode Checks:', suffix = 'progress', length = 50)
	while not (chassis_object.investigation_hubnode_progress >= chassis_object.investigation_hubnode_target) and not (counter_elapsed >= 400) and not chassis_object.investigation_fail_through and not chassis_object.dns_error:
		time.sleep(.5)
		suffix='progress'
		if chassis_object.investigation_hubnode_progress == chassis_object.investigation_hubnode_target:
			suffix="DONE       "
		printProgressBar(chassis_object.investigation_hubnode_progress, chassis_object.investigation_hubnode_target, prefix = 'Hubnode Checks:', suffix = suffix, length = 50)
	if chassis_object.investigation_fail_through: printProgressBar(chassis_object.investigation_hubnode_progress, chassis_object.investigation_hubnode_target, prefix = 'Hubnode Checks:', suffix = 'FALED LOGIN', length = 50)
	if chassis_object.dns_error: printProgressBar(chassis_object.investigation_hubnode_progress, chassis_object.investigation_hubnode_target, prefix = 'Hubnode Checks:', suffix = 'DNS NOT FOUND', length = 50)
	# Loading bar for node checks
	print()
	time.sleep(.5)
	counter_elapsed = 0
	printProgressBar(0, chassis_object.investigation_node_target, prefix = 'Node Checks:   ', suffix = 'progress', length = 50)
	while not (chassis_object.investigation_node_progress >= chassis_object.investigation_node_target) and not (counter_elapsed >= 400) and not chassis_object.investigation_fail_through and not chassis_object.dns_error:
		time.sleep(.5)
		suffix='progress'
		if chassis_object.investigation_node_progress == chassis_object.investigation_node_target:
			suffix="DONE       "
		printProgressBar(chassis_object.investigation_node_progress, chassis_object.investigation_node_target, prefix = 'Node Checks:   ', suffix = suffix, length = 50)
	if chassis_object.investigation_fail_through or chassis_object.dns_error:printProgressBar(chassis_object.investigation_node_progress, chassis_object.investigation_node_target, prefix = 'Node Checks:   ', suffix = 'N/A        ', length = 50)
	print()


def initial_console_SSH_connection(channel, verbose = False):
	tempBuffer = ''
	expected_prompts = {
		"(.*ipmitool>\ +$)": (True, "set targetaddr 0x20"),
		"(.*active-hub1>\ +$)": (True, "set targetaddr 0x20"),
		"(.*backup-hub2>\ +$)": (True, "set targetaddr 0x20"),
		"(.*MSH8900\ +)": (False, "admin"),
		"(.*login:\ +$)": (False, "admin"),
		"(.*Username:\ +$)": (False, "admin"),
		"(.*Password:\ +$)": (False, "admin"),
		"(.*console.\ +$)": (False, "\x0d")
		}
	done = False
	counter = 0
	if verbose: print('\n==================================\n INITIAL SERIAL CONNECTION BUFFER \n==================================')
	while not done:
		counter += 1
		buffer = channel.recv(4096)
		if verbose:	print(buffer.decode("utf-8"))
		tempBuffer += buffer.decode("utf-8")
		for prompt in expected_prompts:
			if re.search(prompt, buffer.decode("utf-8")):

				channel.send(expected_prompts[prompt][1])
				channel.send("\x0d")
				time.sleep(2)
				done = expected_prompts[prompt][0]
		if done == False and counter == 5:
			channel.send("\x0d")
			time.sleep(2)
			channel.send("\x0d")
			counter = 0
	return tempBuffer

def run_IPMI_Command_Console(channel, cmd, verbose = False):
	tempBuffer = ''
	done = False
	channel.send("\x0d")
	channel.send(cmd)
	channel.send("\x0d")
	time.sleep(2)

	done = False
	if verbose: print('\n=======================================\n IOL COMMANDS SERIAL CONNECTION BUFFER \n=======================================')
	while not done:
		buffer = channel.recv(4096)
		if verbose:	print(buffer.decode("utf-8"))
		tempBuffer += buffer.decode("utf-8")
		if re.search("(.*ipmitool>\ +$)", buffer.decode("utf-8")):
			done = True
		elif re.search("(.*active-hub1>\ +$)", buffer.decode("utf-8")):
			done = True
		elif re.search("(.*backup-hub2>\ +$)", buffer.decode("utf-8")):
			done = True
		else:
			channel.send("\x0d")
	time.sleep(2)

	return tempBuffer


def parse_console_output(needle,haystack,exclude = None):
	result=[]
	cleaned = haystack.split('\n')
	for value in cleaned:
		if re.search("(" + needle + ")", value):
			if exclude == None or not re.search("(" + exclude + ")", value):
				value = value.strip().rstrip()
				value = " ".join(value.split())
				value = value.replace(' : ', '\t')
				value = value.replace(' | ', '\t')
				value = value.replace('| ', '')
				value = value.replace('|* ', '')
				value = value.replace(' |', '')
				value = value.replace('|', '\t')
				result.append(value)

	result = [i.split('\t',1)[1] for i in result]
	return result


def hubnode_console_inspect_base(user, passwd, console_target, console_port, verbose = False):

	# General Variables
	channel = None
	result = dict()
	
	# Create SSH session
	ssh = paramiko.SSHClient()
	ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())
	# Try block on connect function for failed credentials
	try:
		ssh.connect(console_target, port=console_port, username=user, password=passwd)
		if verbose: print("Connected to %s for serial inspection" % console_target)
	except paramiko.AuthenticationException:
		if verbose: print("Connection to %s attempted, Error: Incorrect Password" % console_target)
		result['ERROR']=str("Authentication failed, please verify your credentials")
		return result
		exit()

	# create SSH shell
	channel = ssh.invoke_shell()


	# Push commands to navigate serial console connection to hubnode / SHMC
	channel.send("\x07")
	channel.send("0")
	channel.send("\x0d")
	time.sleep(2)


	runningBuffer = ''
	runningBuffer += initial_console_SSH_connection(channel,verbose)

	# Run Commands to get data into buffer output (variable runningBuffer)
	runningBuffer += run_IPMI_Command_Console(channel,'fru',verbose)
	#runningBuffer += run_IPMI_Command_Console(channel,'hpm check',verbose) # Removed in favor of later discovery with IPMI over LAN
	runningBuffer += run_IPMI_Command_Console(channel,'lan print',verbose)
	
	result["Board Serial"]=parse_console_output("Board Serial",runningBuffer)
	result["Board Product"]=parse_console_output("Board Product",runningBuffer)
	result["IP Address"]=parse_console_output("IP Address",runningBuffer,"Source")
	result["Chassis Serial"]=parse_console_output("Chassis Serial",runningBuffer)
	result["MAC Address"]=parse_console_output("MAC Address",runningBuffer)

	# remove duplicates from Chassis Serial response
	result["Chassis Serial"] = list(dict.fromkeys(result["Chassis Serial"]))


	ssh.close()
	if verbose: print("Connection closed to %s" % console_target)

	return result;

def hostname_resolves(hostname):
    try:
        socket.gethostbyname(hostname)
        return 1
    except socket.error:
        return 0

###############################################################################
## Main Test Function #########################################################
###############################################################################

def main():
	# Create and set parser for arguments
	parser = argparse.ArgumentParser(description='This script is for Kontron auditing only')
	parser.add_argument('-s', '--site', metavar='Target-Site', type=str, help='This is the name of the site (aka PoP) where the target device exists, ex: sjc02')
	parser.add_argument('-r', '--rack', metavar='Target-Rack', type=str, help='This is the number designation of the rack that contains the target device, ex: 217')
	parser.add_argument('-c', '--chassis', metavar='Target-Chassis', type=str, help='This is the name of the rack that contains the target device, ex: r217')
	parser.add_argument('-v', '--verbose', action="store_true", default=False, help='Verbose Output; Print entire SSH buffer to stdout while script is running')
	parser.add_argument('-u', dest='user', type=str, help='Username for authentication to serial terminal server')
	parser.add_argument('-p', dest='passwd', type=str, help='Password for authentication to serial terminal server - FYI: cleartext passwords are bad')
	parser.add_argument('-ni', dest='noninteractive', action="store_true", default=False, help='Disable credentials prompt (use with SSH keys)')

	args = parser.parse_args()
	print()
	if args.verbose:
		print('Verbose Mode Enabled')
		print('\n',str(args),'\n')

	CONSOLE_username = ''
	CONSOLE_password = ''

	curr_user = getpass.getuser()
	run_state = 0
	if args.noninteractive:
		run_state = 1
	if args.noninteractive and curr_user == 'root':
		print('Non-interactive mode not supported for root user')
		run_state = 2
	
	if run_state == 1:
		CONSOLE_username = curr_user
		CONSOLE_password = ''
	if run_state != 1:
		if args.user:
			CONSOLE_username = args.user
		else:
			CONSOLE_username = input("Serial Terminal SSH Username: ")
		if args.passwd:
			CONSOLE_password = args.passwd
		else:
			CONSOLE_password = getpass.getpass()
		print()


	testChassis = kontronChassis(args.site,args.rack,args.chassis)

	# Create loading bar thread
	if not args.verbose: testChassis.investigation_loading_bar=threading.Thread(target=loading_bar, args=(testChassis,))
	if not args.verbose: testChassis.investigation_loading_bar.start()
	testChassis.investigate(CONSOLE_username,CONSOLE_password,args.verbose)
	if not args.verbose: testChassis.investigation_loading_bar.join()
	
	print('\n\n\n')
	print(testChassis.printTable())
	print('\n\n\n')

	print('Warnings:')
	warnings = testChassis.check()
	for key,values in sorted(warnings.items()):
		print('    '+str(key))
		for value in values:
			print("        "+str(value))
	print()

if __name__ == "__main__":
	main()
