/*
 * Kontron Canada, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */

/*** DESCRIPTION SECTION ***************************************************************************
*
*  File Name:     vkcs.c
*  Creation Date: 2010-01-20
*  Programmer:    Marie-Josee Blais
*  Description:   This is the vkcs (Virtual KCS) ipmitool plugin
*
***************************************************************************************************/

/****************************************/
/** include files **/
/****************************************/
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/helper.h>
#include <ipmitool/log.h>
#include <ipmitool/ipmi_cc.h>

#if defined(HAVE_CONFIG_H)
# include <config.h>
#endif

# if defined(HAVE_LINUX_COMPILER_H)
#  include <linux/compiler.h>
# endif

# include <linux/ipmi.h>

/****************************************/
/** private definitions and macros **/
/****************************************/
#define VKCS_MAX_DATASIZE        40

/* ioctl cmd definition -- Must match viface.h definition */
#define VKCS_WRITE_READ_KCS_MESSAGE _IOWR( 'V', 5, tVkcsTxStruct )

#define VKCS_DEV                 "/dev/viface"

#define VKCS_IFACE_KCS_CLIENT    ((unsigned char)1)

typedef struct sVkcsTxStruct
{
   unsigned char  len;
   unsigned char  ifaceInst;
   unsigned char  unusedPack0;
   unsigned char  unusedPack1;
   unsigned char  buf[VKCS_MAX_DATASIZE];
} tVkcsTxStruct;

#define IPMI_KCS_HEADER_SIZE  3
#define IPMI_IPMB_HEADER_SIZE 8

extern int verbose;

/****************************************/
/** private functions declaration **/
/****************************************/
static int ipmi_vkcs_open(struct ipmi_intf * intf);
static void ipmi_vkcs_close(struct ipmi_intf * intf);
static struct ipmi_rs * ipmi_vkcs_send_cmd(struct ipmi_intf * intf, struct ipmi_rq * req);
static int send_to_vkcs_driver(int fd, tVkcsTxStruct* txStruct);
static void print_data_buffer(int verbose_level, char *title, tVkcsTxStruct* txStruct);

/************************************************
*
* Function Name: ipmi_vkcs_open
*
* Description: Opens a filehandle to the vkcs device node
*
* Restriction: None
*
* Input: intf: Pointer on a structure that holds sub-function data
*              about the IPMI command
*
* Output: intf->fd: vkcs filehandle
*         intf->opened: Filehandle state
*
* Global: None
*
* Return: vkcs Filehandle also used to test if device was opened.
*
************************************************/
static int ipmi_vkcs_open(struct ipmi_intf * intf)
{
	lprintf(LOG_INFO, "ipmi_vks_open called !\n");
	intf->fd = open(VKCS_DEV, O_RDWR, 0);
	if (intf->fd >= 0)
	{
		intf->opened = 1;
	}

	return intf->fd;
}

/************************************************
*
* Function Name: ipmi_vkcs_close
*
* Description: Closes the filehandle to the vkcs device node
*
* Restriction: None
*
* Input: intf: Pointer on a structure that holds sub-function data
*              about the IPMI command
*
* Output: intf->fd: vkcs filehandle
*         intf->opened: Filehandle state
*
* Global: None
*
* Return: None
*
************************************************/
static void ipmi_vkcs_close(struct ipmi_intf * intf)
{
	if (intf->fd >= 0)
	{
		close(intf->fd);
	}
	intf->fd = -1;
	intf->opened = 0;
}

/************************************************
*
* Function Name: ipmi_vkcs_send_cmd
*
* Description: This function receives the ipmi command and formats
*              it for vkcs
*
* Restriction: None
*
* Input: intf:
*        req:
*
* Output: None
*
* Global: None
*
* Return: ipmi_rs pointer: Pointer to receive data
*
* vkcs driver data example: <Data length> <Data 0> ... <Data N>
*
************************************************/
static struct ipmi_rs * ipmi_vkcs_send_cmd(struct ipmi_intf * intf, struct ipmi_rq * req)
{
   static tVkcsTxStruct* txStruct;
	int data_len = 0;
	uint8_t index;
	static struct ipmi_rs rsp;

	/* Open device handle if necessary and return NULL if open fails */
	if (intf->opened == 0 && intf->open != NULL && intf->open(intf) < 0)
	{
		return NULL;
	}
   
   txStruct = malloc(sizeof(tVkcsTxStruct));
   if( txStruct == NULL )
   {
      printf("vkcs: could not allocate memory\n");
      return NULL;
   }

	index = 0;
   memset((unsigned char*)txStruct, 0, sizeof(tVkcsTxStruct));
	memset(rsp.data, 0, VKCS_MAX_DATASIZE);
   
   txStruct->ifaceInst = VKCS_IFACE_KCS_CLIENT;

	/* KCS */
	if (intf->transit_addr == 0 && (intf->target_addr == 0 || intf->target_addr	== intf->my_addr))
	{
		if (req->msg.data_len <= VKCS_MAX_DATASIZE - 3)
		{
			/* Add length of message */
         txStruct->len = req->msg.data_len + 2; /* +2: NetFn + cmd */
			/* Add Netfn/LUN */
         txStruct->buf[index++] = ((req->msg.netfn << 2) | (intf->target_lun & 0x03));
			/* Add command */
         txStruct->buf[index++] =req->msg.cmd;
			/* Add data */
			memcpy( txStruct->buf + index, req->msg.data, req->msg.data_len);
			
			print_data_buffer(LOG_DEBUG, "VKCS KCS Send", txStruct);
			rsp.ccode = send_to_vkcs_driver(intf->fd, txStruct);
			print_data_buffer(LOG_DEBUG, "VKCS KCS Receive", txStruct);

			if (txStruct->len < VKCS_MAX_DATASIZE )
			{
				rsp.ccode = txStruct->buf[2];
				rsp.data_len = txStruct->len - 3;
				memcpy(rsp.data, txStruct->buf + 3, rsp.data_len);
			}
			else
			{
				rsp.ccode = IPMI_CC_CANT_RET_NUM_REQ_BYTES;
				rsp.data_len = 0;
			}
		}
		else
		{
			rsp.ccode = IPMI_CC_REQ_DATA_FIELD_EXCEED;
			rsp.data_len = 0;
		}
	}
	/* IPMB */
	else if (intf->transit_addr == 0 || intf->transit_addr == intf->my_addr)
	{
		/* Add length of message */
      txStruct->len = req->msg.data_len + 10;
		/* Add Netfn/LUN */
      txStruct->buf[index++] = ((IPMI_NETFN_APP << 2) | (0x00 & 0x03));
		/* Add "Send" command */
		txStruct->buf[index++] = IPMI_APP_SEND_MESSAGE;
		/* Add channel + Track request*/
		txStruct->buf[index++] = intf->target_channel | 0x40;
		if(intf->broadcast == TRUE && req->msg.netfn == 0x06 && req->msg.cmd == 0x01)
		{
			txStruct->buf[index++] = 0x00;
			lprintf(LOG_DEBUG, "broadcast enabled");
		}
		/* Add Responder address */
		txStruct->buf[index++] = intf->target_addr;
		/* Add Netfn/LUN */
		txStruct->buf[index++] = ((req->msg.netfn << 2) | (intf->target_lun & 0x03));
		/* Add Checksum */
		txStruct->buf[index] = ipmi_csum(txStruct->buf + index - 2, 2);
		index++;
		/* Add Requester address */
		txStruct->buf[index++] = intf->my_addr;
		/* Add Requester seq/Lun (LUN 2 is used by vkcs)*/
		txStruct->buf[index++] = ((0x00 << 2) | (0x02 & 0x03));
		/* Add command */
		txStruct->buf[index++] = req->msg.cmd;
		/* Add data */
		memcpy( txStruct->buf + index, req->msg.data, req->msg.data_len);
		index += req->msg.data_len;
		/* Add Checksum */
		txStruct->buf[index] = ipmi_csum(txStruct->buf + index - 3 - req->msg.data_len, 3 + req->msg.data_len);
		print_data_buffer(LOG_DEBUG, "VKCS IPMB Send", txStruct);
		rsp.ccode = send_to_vkcs_driver(intf->fd, txStruct);
		print_data_buffer(LOG_DEBUG, "VKCS IPMB Receive", txStruct);
		rsp.ccode = txStruct->buf[2];
		rsp.data_len = txStruct->len - IPMI_KCS_HEADER_SIZE;

		if( txStruct->len < VKCS_MAX_DATASIZE)
		{
			if (rsp.ccode == 0)
			{
			    /* Decapsulate command from send command */
				rsp.data_len -= IPMI_IPMB_HEADER_SIZE;
				rsp.ccode = txStruct->buf[9];
				memcpy(rsp.data, txStruct->buf + 10, rsp.data_len);
			}
			else
			{
				rsp.data_len = 0;
			}

		}
		else
		{
			rsp.ccode = IPMI_CC_REQ_DATA_FIELD_EXCEED;
			rsp.data_len = 0;
		}
	}
	/* Double Bridging not supported for now*/
	else
	{
		rsp.ccode = IPMI_CC_DESTINATION_UNAVAILABLE;
		rsp.data_len = 0;
	}
   
   free( txStruct );

	return &rsp;
}

/************************************************
*
* Function Name: send_to_vkcs_driver
*
* Description: Sends data to vkcs driver
*
* Restriction: None
*
* Input: fd: vkcs filehandle
*        buffer: Pointer to data buffer to send to vkcs driver
*
* Output: buffer: Pointer to data buffer to receive from vkcs driver
*
* Global: None
*
* Return: Return code from vkcs driver
*
* vkcs driver data example: <Data length> <Data 0> ... <Data N>
*
************************************************/
static int send_to_vkcs_driver(int fd, tVkcsTxStruct *txStruct)
{
	int returnCode = 0;

	returnCode = ioctl(fd, VKCS_WRITE_READ_KCS_MESSAGE, (unsigned int *) txStruct);

	if (returnCode == 0 && txStruct->len > 0)
	{
		returnCode = txStruct->buf[0];
	}
	else
	{
		printf("VKCS Driver Error:%02X\r\n", returnCode);
		txStruct->buf[0] = IPMI_CC_UNSPECIFIED_ERROR;
		txStruct->len = 3;
		returnCode = IPMI_CC_UNSPECIFIED_ERROR;
	}

	return returnCode;
}

/************************************************
*
* Function Name: print_data_buffer
*
* Description: Prints data sent to or received from vkcs driver 
*
* Restriction: None
*
* Input: verbose_level: Verbose level needed to print
         title: Title written to identify data
         buffer: Pointer to data buffer
*
* Output: None
*
* Global: None
*
* Return: None
*
************************************************/
static void print_data_buffer(int verbose_level, char *title, tVkcsTxStruct* txStruct)
{
	uint8_t index = 0;
	
	if (verbose > verbose_level)
	{
		printf("%s:", title);
		for (index = 0; index <= txStruct->len; index++)
		{
			printf("%X ", txStruct->buf[index]);
		}
		printf("\n");
	}
}

/* Function pointer structure used by the plugin system */
struct ipmi_intf ipmi_vkcs_intf =
{
        name: "vkcs",
        desc: "Linux VKCS Interface",
        open: ipmi_vkcs_open,
        close: ipmi_vkcs_close,
        sendrecv: ipmi_vkcs_send_cmd,
        my_addr: IPMI_BMC_SLAVE_ADDR,
        target_addr: IPMI_BMC_SLAVE_ADDR,
	channel_buf_size: 0
};
