package main

import (
	"context"
	"log"
	"os"

	"code.justin.tv/dcops/rack_buildsheets/pkg/rackcheck"
	"github.com/urfave/cli"
)

const buildsheetServiceURLFlag = "buildsheet-service-url"
const rebuildPathFlag = "rebuild-path"

func main() {
	app := cli.NewApp()
	app.Name = "rackcheck"
	app.Description = "cli tooling for git-based buildsheet validation/syncing"
	app.Usage = app.Description
	app.UsageText = "rackcheck [command] [flags]"
	app.Version = "0.2.0"

	flags := []cli.Flag{
		cli.StringFlag{
			Name:  buildsheetServiceURLFlag,
			Value: "https://buildsheet-service.us-east-2.staging.provisioner.live-video.a2z.com",
			Usage: "Upstream url for the buildsheet service",
		},
	}

	app.Commands = []cli.Command{
		{
			Name:  "validate",
			Usage: "validate a set of changed buildsheets against buildsheet-service",
			Action: func(c *cli.Context) error {
				if err := rackcheck.Validate(context.Background(), c.String(buildsheetServiceURLFlag)); err != nil {
					log.Fatal(err)
				}
				return nil
			},
			Flags: flags,
		},
		{
			Name:  "sync",
			Usage: "sync a set of changed buildsheets to buildsheet-service",
			Action: func(c *cli.Context) error {
				if err := rackcheck.Sync(context.Background(), c.String(buildsheetServiceURLFlag)); err != nil {
					log.Fatal(err)
				}
				return nil
			},
			Flags: flags,
		},
		{
			Name:  "health",
			Usage: "check connectivity to the buildsheet service",
			Action: func(c *cli.Context) error {
				if err := rackcheck.Health(context.Background(), c.String(buildsheetServiceURLFlag)); err != nil {
					log.Fatal(err)
				}
				return nil
			},
			Flags: flags,
		},
		{
			Name:  "rebuild",
			Usage: "delete all known files in the git repo and resync them all in batches",
			Action: func(c *cli.Context) error {
				if err := rackcheck.Rebuild(
					context.Background(),
					c.String(buildsheetServiceURLFlag),
					c.String(rebuildPathFlag),
					c.Int("goroutines"),
				); err != nil {
					log.Fatal(err)
				}
				return nil
			},
			Flags: []cli.Flag{
				flags[0],
				cli.StringFlag{
					Name:  rebuildPathFlag,
					Value: "./pop",
					Usage: "optional path to recursively search when rebuilding from git",
				},
				cli.IntFlag{
					Name:  "goroutines",
					Value: 1,
					Usage: "optional number of goroutines to use to rebuild",
				},
			},
		},
	}

	if err := app.Run(os.Args); err != nil {
		log.Fatal(err)
	}
}
