package differ

import (
	"context"
	"fmt"
	"io/ioutil"
	"os"
	"os/exec"
	"path/filepath"
	"regexp"
	"strings"

	"code.justin.tv/dcops/rack_buildsheets/pkg/spec"
	bsSpec "code.justin.tv/video-tools/buildsheet-service/pkg/spec"
	"github.com/pkg/errors"
)

const headMinusOneTarget = "HEAD~1"
const masterTarget = "origin/master"

// Differ abstracts the introspection of changed buildsheets in git
type Differ struct {
	Filer  spec.Filer
	Gitter spec.Gitter
}

// NewSyncDiffer returns a new Differ configured to operate on sync and validate calls
func NewSyncDiffer(ctx context.Context) *Differ {
	return &Differ{
		Filer:  &rackCheckFiler{},
		Gitter: &syncGitter{},
	}
}

// NewRebuildDiffer returns a new Differ configured to operate on rebuild calls
func NewRebuildDiffer(ctx context.Context, rebuildPath string) *Differ {
	return &Differ{
		Filer:  &rackCheckFiler{},
		Gitter: &rebuildGitter{path: rebuildPath},
	}
}

// GetChangedBuildSheets returns a slice of ChangedBuildSheet objects representing git changes
// since the reference point (as defined by current branch)
func (d *Differ) GetChangedBuildSheets(ctx context.Context) (bsSpec.BulkBuildSheetData, error) {
	isMaster, err := d.Gitter.CheckIsMaster()
	if err != nil {
		return nil, errors.Wrap(err, "checking if master")
	}

	var diffTarget string
	if isMaster {
		diffTarget = headMinusOneTarget
	} else {
		diffTarget = masterTarget
	}

	diffLines, err := d.Gitter.NameStatusDiff(diffTarget)
	if err != nil {
		return nil, errors.Wrap(err, "getting diffs from gitter")
	}

	var (
		changes bsSpec.BulkBuildSheetData
		change  *bsSpec.ChangedBuildSheet
	)

	rg, err := regexp.Compile(spec.ChangedBuildSheetRegexp)
	if err != nil {
		return nil, errors.Wrap(err, "compiling changed buildsheet regex")
	}

	for _, line := range strings.Split(string(diffLines), "\n") {
		if !rg.MatchString(line) {
			continue
		}

		change, err = spec.NewChangedBuildSheet(line, d.Filer)
		if err != nil {
			return nil, errors.Wrapf(err, "parsing buildsheet from line: %v", line)
		}

		changes = append(changes, change)
	}

	return changes, nil
}

type rackCheckFiler struct{}

func (f *rackCheckFiler) ReadFile(filename string) ([]byte, error) {
	return ioutil.ReadFile(filename)
}

type syncGitter struct{}

func (g *syncGitter) CheckIsMaster() (bool, error) {
	// check for the branch_name var to be set, which assumes jenkins
	branchName := os.Getenv("BRANCH_NAME")
	if branchName != "" {
		return branchName == "master", nil
	}

	// if that didn't trigger, you're probably local. this will work
	// unless you have a detached head, which is probably not where you want to be
	brancher := exec.Command("git", "rev-parse", "--abbrev-ref", "HEAD")
	out, err := brancher.CombinedOutput()
	if err != nil {
		return false, errors.New(string(out))
	}

	return string(out) == "master\n", nil
}

func (g *syncGitter) NameStatusDiff(diffTarget string) ([]byte, error) {
	diffFiles := exec.Command("git", "diff", "--name-status", diffTarget)
	out, err := diffFiles.CombinedOutput()
	if err != nil {
		return nil, errors.New(string(out))
	}

	return out, nil
}

// rebuildGitter is a convenience implementation of the Gitter interface that
// will use its filer to return "Modified" changed buildsheets for all known
// buildsheets in the git repo
type rebuildGitter struct {
	path string
}

// CheckIsMaster doesn't do anything, and is implemented to satisfy the interface
func (g *rebuildGitter) CheckIsMaster() (bool, error) {
	return false, nil
}

// NameStatusDiff walks all buildsheets and builds a byte array which appears to the caller
// as though every known buildsheet in the repo has been modified
func (g *rebuildGitter) NameStatusDiff(diffTarget string) ([]byte, error) {
	rg, err := regexp.Compile(spec.BuildSheetFileRegexp)
	if err != nil {
		return nil, errors.Wrap(err, "compiling build sheet file regexp")
	}

	var buildSheetPaths []string
	err = filepath.Walk(g.path, func(path string, info os.FileInfo, err error) error {
		if rg.Match([]byte(path)) {
			gitifiedPath := fmt.Sprintf("M\t%s", path)
			buildSheetPaths = append(buildSheetPaths, gitifiedPath)
		}
		return nil
	})
	if err != nil {
		return nil, err
	}

	return []byte(strings.Join(buildSheetPaths, "\n")), nil
}
