package differ_test

import (
	"context"
	"strings"
	"testing"

	"code.justin.tv/dcops/rack_buildsheets/pkg/differ"
	"code.justin.tv/dcops/rack_buildsheets/pkg/spec"
	"github.com/stretchr/testify/require"
)

const deletedBuildSheetString = "D\tpop/sjc02/rack/r202/RU02/garbage.json"
const garbageBuildSheetString = "Z\tpop/sjc02/rack/r202/RU01/garbage.json"
const modifiedBuildSheetString = "M\tpop/sjc02/rack/r202/RU03/garbage.json"

func TestGetChangedBuildSheets(t *testing.T) {
	changedFileStrings := []string{
		"A\t.DS_Store",
		modifiedBuildSheetString,
		deletedBuildSheetString,
		garbageBuildSheetString,
	}

	gitter := &testGitter{
		outputLines: changedFileStrings,
	}

	differ := makeTestDiffer(t, gitter)

	// check out master changes
	gitter.isMaster = true
	changes, err := differ.GetChangedBuildSheets(context.Background())
	require.NoError(t, err)
	require.Equal(t, "HEAD~1", gitter.receivedTarget)

	require.Len(t, changes, 2)
	rackUnits := make([]int, 2)
	for i, ch := range changes {
		rackUnits[i] = ch.RackUnit
	}

	require.Contains(t, rackUnits, 2, 3)
	require.NotContains(t, rackUnits, 1)

	// make sure that we check master if we're on a different branch
	gitter.isMaster = false
	_, err = differ.GetChangedBuildSheets(context.Background())
	require.NoError(t, err)
	require.Equal(t, "origin/master", gitter.receivedTarget)
}

func makeTestDiffer(t *testing.T, gitter spec.Gitter) *differ.Differ {
	return &differ.Differ{
		Filer:  &testFiler{},
		Gitter: gitter,
	}
}

type testGitter struct {
	isMaster       bool
	outputLines    []string
	receivedTarget string
}

func (t *testGitter) CheckIsMaster() (bool, error) {
	return t.isMaster, nil
}

func (t *testGitter) NameStatusDiff(target string) ([]byte, error) {
	t.receivedTarget = target
	return []byte(strings.Join(t.outputLines, "\n")), nil
}

type testFiler struct{}

func (t *testFiler) ReadFile(filename string) ([]byte, error) {
	return []byte(""), nil
}
