package rackcheck

import (
	"context"
	"fmt"
	"log"
	"net/http"
	"strings"
	"sync"
	"time"

	"code.justin.tv/dcops/rack_buildsheets/pkg/differ"
	"code.justin.tv/video-tools/buildsheet-service/api/buildsheet"
	bsspec "code.justin.tv/video-tools/buildsheet-service/pkg/spec"
	"github.com/pkg/errors"
)

func buildsheetServiceClient(url string) buildsheet.BuildSheetService {
	log.Printf("creating buildsheet service client at url: %s", url)
	timeoutClient := &http.Client{Timeout: 10 * time.Minute}
	return buildsheet.NewBuildSheetServiceProtobufClient(url, timeoutClient)
}

// Validate gathers buildsheet changes and validates them against the buildsheet service
func Validate(ctx context.Context, serviceURL string) error {
	d := differ.NewSyncDiffer(ctx)
	changes, err := d.GetChangedBuildSheets(ctx)
	if err != nil {
		return err
	}

	if len(changes) == 0 {
		log.Println("no changes")
		return nil
	}

	protoChanges, err := bsspec.PBChangedBuildSheetsFromChangedBuildSheets(ctx, changes)
	if err != nil {
		return err
	}

	client := buildsheetServiceClient(serviceURL)
	if _, err := client.Validate(ctx, &buildsheet.ValidateRequest{
		ChangedBuildSheets: protoChanges,
	}); err != nil {
		return err
	}

	return nil
}

// Sync gathers buildsheet changes and syncs them to the buildsheet service
func Sync(ctx context.Context, serviceURL string) error {
	d := differ.NewSyncDiffer(ctx)
	changes, err := d.GetChangedBuildSheets(ctx)
	if err != nil {
		return err
	}

	if len(changes) == 0 {
		log.Println("no changes")
		return nil
	}

	protoChanges, err := bsspec.PBChangedBuildSheetsFromChangedBuildSheets(ctx, changes)
	if err != nil {
		return err
	}

	client := buildsheetServiceClient(serviceURL)
	if _, err := client.Sync(ctx, &buildsheet.SyncRequest{
		ChangedBuildSheets: protoChanges,
	}); err != nil {
		return err
	}

	return nil
}

// Health pings the service endpoint
func Health(ctx context.Context, serviceURL string) error {
	client := buildsheetServiceClient(serviceURL)
	if _, err := client.Health(ctx, &buildsheet.HealthRequest{}); err != nil {
		return err
	}

	return nil
}

// Rebuild gathers all buildsheets under a given path (relative to ./pop) and
// syncs them to the buildsheet-service in batches of 8
func Rebuild(ctx context.Context, serviceURL, rebuildPath string, goroutineCount int) error {
	client := buildsheetServiceClient(serviceURL)
	batchSize := 8
	d := differ.NewRebuildDiffer(ctx, rebuildPath)

	allChanges, err := d.GetChangedBuildSheets(ctx)
	if err != nil {
		return err
	}

	log.Printf("Converting buildsheets")
	protoChanges, err := bsspec.PBChangedBuildSheetsFromChangedBuildSheets(ctx, allChanges)
	if err != nil {
		return errors.Wrap(err, "converting raw buildsheets to proto representation")
	}

	allChangesLength := len(allChanges)
	log.Printf("Replacing %d buildsheets using %d goroutines", allChangesLength, goroutineCount)
	goroutineBatchSize := allChangesLength / goroutineCount
	var wg sync.WaitGroup
	for goNum := 0; goNum < goroutineCount; goNum++ {
		wg.Add(1)
		go func(num int, batch []*buildsheet.ChangedBuildSheet) {
			defer wg.Done()
			var changeBatch []*buildsheet.ChangedBuildSheet
			for len(batch) > 0 {
				if len(batch) < batchSize {
					changeBatch, batch = batch, nil
				} else {
					changeBatch, batch = batch[:batchSize], batch[batchSize:]
				}

				log.Printf("[goroutine %d] Syncing %d buildsheets, %d remain...", num, len(changeBatch), len(batch))
				if _, err := client.Sync(ctx, &buildsheet.SyncRequest{ChangedBuildSheets: changeBatch}); err != nil {
					locations := make([]string, len(changeBatch))
					for i, bs := range changeBatch {
						loc := bs.GetLocation()
						locations[i] = fmt.Sprintf("%v/%v/%d", loc.Pop, loc.Rack, loc.RackUnit)
					}
					log.Printf("[goroutine %d] unrecoverable error somewhere in the following buildsheets:", num)
					log.Printf("[goroutine %d] %s", num, strings.Join(locations, ", "))
					log.Printf("[goroutine %d] %s", num, err)
					break
				}
			}
			if batch == nil || len(batch) == 0 {
				log.Printf("[goroutine %d] changes synced", num)
			}
		}(goNum, protoChanges[goNum*goroutineBatchSize:goroutineBatchSize*(goNum+1)])
	}

	wg.Wait()

	return nil
}
