package spec

import (
	"strconv"
	"strings"

	bsSpec "code.justin.tv/video-tools/buildsheet-service/pkg/spec"
	"github.com/pkg/errors"
)

// BuildSheetFileRegexp matches lines in filepath.Walk output corresponding to buildsheets
const BuildSheetFileRegexp = `^pop\/\D{3}\d{2}\/rack\/r\d{3}\/RU\d{2}\/\S*\.json$`

// ChangedBuildSheetRegexp matches lines in git diff output corresponding to changed buildsheets
const ChangedBuildSheetRegexp = `^[ADM]\tpop\/\D{3}\d{2}\/rack\/r\d{3}\/RU\d{2}\/\S*\.json$`

const gitComponentsLen = 2
const gitChangeTypeComponent = 0
const gitChangeTypeDeleted = "D"

const pathComponentsLen = 6
const popComponent = 1
const rackComponent = 3
const rackUnitComponent = 4

// NewChangedBuildSheet takes a line from git diff output and returns a ChangedBuildSheet
func NewChangedBuildSheet(line string, filer Filer) (*bsSpec.ChangedBuildSheet, error) {
	gitComponents := strings.Split(line, "\t")
	if len(gitComponents) != gitComponentsLen {
		return nil, errors.Errorf(
			"expected %d tab-separated git components in line: %v",
			gitComponentsLen,
			line,
		)
	}

	pathComponents := strings.Split(gitComponents[1], "/")
	if len(pathComponents) != pathComponentsLen {
		return nil, errors.Errorf(
			"expected %d \"/\"-separated path components in line: %v",
			pathComponentsLen,
			line,
		)
	}

	rawRackUnit := strings.ReplaceAll(pathComponents[rackUnitComponent], "RU", "")
	rackUnit, err := strconv.Atoi(rawRackUnit)
	if err != nil {
		return nil, errors.Errorf("could not parse rack unit from line: %v", line)
	}

	var data []byte
	// only populate data for modified or added buildsheets
	if gitComponents[gitChangeTypeComponent] != gitChangeTypeDeleted {
		data, err = filer.ReadFile(gitComponents[1])
		if err != nil {
			return nil, errors.Wrap(err, "reading buildsheet data")
		}
	}

	return &bsSpec.ChangedBuildSheet{
		ChangeType: gitComponents[gitChangeTypeComponent],
		Data:       data,
		Pop:        pathComponents[popComponent],
		Rack:       pathComponents[rackComponent],
		RackUnit:   rackUnit,
	}, nil
}
