package spec

import (
	"crypto/md5"
	"encoding/hex"
	"encoding/json"
	"strings"
	"time"

	"github.com/pkg/errors"
)

// BuildSheet represents a master buildsheet that defines the contents of a Kontron Chassis
type BuildSheet struct {
	components     []component
	pop            string
	rack           string
	rackUnit       int
	commitDateTime time.Time
	commitHash     string
}

type component map[string]string

// NewBuildSheet accepts json bytes and returns a BuildSheet object
func NewBuildSheet(data []byte, pop, rack string, rackUnit int, commitTime time.Time) (*BuildSheet, error) {
	bs := &BuildSheet{
		pop:      pop,
		rack:     rack,
		rackUnit: rackUnit,
	}

	if len(data) == 0 || string(data) == "{}" {
		return bs, nil
	}

	var rawComponents []json.RawMessage
	if err := json.Unmarshal(data, &rawComponents); err != nil {
		return nil, errors.Wrap(err, "parsing buildsheet bytes")
	}

	components := make([]component, len(rawComponents))
	for i, rC := range rawComponents {
		if err := json.Unmarshal(rC, &components[i]); err != nil {
			return nil, errors.Wrapf(err, "parsing component: %v", rC)
		}
	}

	bs.components = components
	bs.commitDateTime = commitTime

	commitHasher := md5.New()
	if _, err := commitHasher.Write(data); err != nil {
		return nil, errors.Wrapf(err, "commit hasher unable to write")
	}
	bs.commitHash = hex.EncodeToString(commitHasher.Sum(nil))

	return bs, nil
}

// ParseChassis populates a chassis record from a BuildSheet
func (bs *BuildSheet) ParseChassis() (*Chassis, error) {
	var (
		parsedChassis  *Chassis
		parsedHubNodes []*HubNode
		parsedNodes    []*Node
		err            error
	)

	if len(bs.components) == 0 {
		return &Chassis{
			Pop:      bs.pop,
			Rack:     bs.rack,
			RackUnit: bs.rackUnit,
		}, nil
	}

	for i, component := range bs.components {
		componentName := component["Component"]
		switch {
		case isChassisComponent(componentName):
			if parsedChassis != nil {
				return nil, errors.Errorf(
					"multiple chassis defined in buildsheet, %v, %v",
					parsedChassis,
					component,
				)
			}
			parsedChassis, err = newChassis(component, bs.pop, bs.rack, bs.rackUnit, bs.commitDateTime, bs.commitHash)
			if err != nil {
				return nil, errors.Wrap(err, "parsing chassis from buildsheet")
			}
		case isHubNodeComponent(componentName):
			// pass i for slot since 0 is skipped, to properly place the hubnodes
			hubNode, err := newHubNode(component, i)
			if err != nil {
				return nil, errors.Wrap(err, "parsing hubnode from buildsheet")
			}
			parsedHubNodes = append(parsedHubNodes, hubNode)
		case isNodeComponent(componentName):
			// pass i-2 for slot, to properly place the nodes
			node, err := newNode(component, i-2)
			if err != nil {
				return nil, errors.Wrap(err, "parsing node from buildsheet")
			}
			parsedNodes = append(parsedNodes, node)
		}
	}

	if parsedChassis == nil || parsedChassis.SerialNumber == 0 {
		return nil, errors.New("no chassis defined in buildsheet")
	}

	parsedChassis.HubNodes = parsedHubNodes
	parsedChassis.Nodes = parsedNodes
	parsedChassis.Pop = bs.pop
	parsedChassis.Rack = bs.rack
	parsedChassis.RackUnit = bs.rackUnit

	if err := parsedChassis.ValidateChildren(); err != nil {
		return nil, errors.Wrap(err, "failed to validate children of Chassis")
	}

	return parsedChassis, nil
}

func isChassisComponent(component string) bool {
	for _, val := range []string{KontronChassisMS2900} {
		if strings.EqualFold(component, val) {
			return true
		}
	}

	return false
}

func isHubNodeComponent(component string) bool {
	for _, val := range []string{KontronHubNodeMSH8900} {
		if strings.EqualFold(component, val) {
			return true
		}
	}

	return false
}

func isNodeComponent(component string) bool {
	for _, val := range []string{KontronNodeMSP8022, KontronNodeMSP805X} {
		if strings.EqualFold(component, val) {
			return true
		}
	}

	return false
}
